﻿;//-----------------------------------------------------------------------------------------------------------------
;// TransColorWindow.ahk[UTF-8] (2026/02/10 14:00)
;// for AutoHotKey v2.0.21 or later
;//
;// http://coltpythonkingcobra.g1.xrea.com/pseudoTweetdeck/
;//                                             @PseudoTwDk
;//-----------------------------------------------------------------------------------------------------------------
;//
;
; フルパスで指定されるウィンドウの透明度や透過色を変更します。
;    TransColorWindow.exe Color Target (AllowStyle) (DenyStyle) (DispError) (Debug)
;       Color
;          0xAARRGGBB : AA=不透明度(00-FF) RR=赤成分(00-FF) GG=緑成分(00-FF) BB=青成分(00-FF)
;          none/off   : 不透明色及び不透明度を無効にします
;       Target
;          ウィンドウ文字列/ahk_exe フルパス/ahk_class クラスID など両端""でAutoHotKey
;          独自のウィンドウ指定記述、現在アクティブなウィンドウを指定する場合は"A"
;          https://ahkscript.github.io/ja/docs/v2/misc/WinTitle.htm
;          対象ウィンドウのウィンドウスタイルは WS_VISIBLE ビットが有効で WS_DISABLED ビットが無効の
;          ものとしています。複数ウィンドウに該当する場合はエラー表示し処理しません。
;          マッチを正規表現で記述したい時は先頭に"REGEX:"を書きます
;          AutoHotKeyV2の正規表現文法は下記参照
;          https://ahkscript.github.io/ja/docs/v2/misc/RegEx-QuickRef.htm
;          ex) "ahk_exe C:\Program Files\Mozilla Firefox\firefox.exe"
;          ex) "ahk_exe C:\Program Files\Vivaldi\Application\vivaldi.exe ahk_class Chrome_WidgetWin_1"
;       AllowStyle
;          処理対象であるウィンドウスタイルのビット積(andをとってマスク自身になれば対象)の16進表記
;          省略された場合やdef/defaultと書かれた時は 0x10000000(WS_VISIBLE) が指定されたものとします
;          全てを対象外にしてしまう 0x00000000 は指定できません
;       DenyStyle
;          処理対象から除外するウィンドウスタイルのビット積(andをとって0になれば対象)の16進表記
;          省略された場合やdef/defaultと書かれた時は 0x08000000(WS_DISABLED) が指定されたものとします
;          全てを対象外にしてしまう 0xFFFFFFFF は指定できません
;       DispError
;          1/on/disp : 対象ウィンドウが見つからない時や複数ヒットした際のエラーダイアログを出す
;          [any] : 上記以外が指定されるか無指定ならエラー時にダイアログを出さない
;       DebugMode
;          1/on/debug : デバッグモードON
;          [any] : 上記以外が指定されるか無指定ならデバッグモードOFF
;
;

/* コマンドオプション数チェック */
if( A_Args.Length < 2 ){
	DispUsage()
}

/* 同時に複数実行を許可する */
#SingleInstance Off

/* 非表示になっているアプリも対象とする（仮想デスクトップ上では非表示扱い？） */
#WinActivateForce
DetectHiddenWindows True

#Include AhkCommon.ahk


/* 変数初期化 */
count := 0
iDisp := 0
iDebug := 0
iRetCode := 1
iAllowStyle := 0
iDenyStyle := 0xFFFFFFFF

colorValue := StrUpper(A_Args[1])
if( StrUpper(colorValue) = "NONE" || StrUpper(colorValue) = "OFF" ){
	colorValue := ""
	colorCode := ""
}
if( StrUpper(SubStr(colorValue,1,2)) = "0X" ){
	colorCode := SubStr(colorValue,5)
	colorValue := SubStr(colorValue,3,2)
}
strTarget := StrReplace(A_Args[2], '"', "")
arg_regx := 2
if( StrUpper(SubStr(strTarget,1,6)) = "REGEX:" ){
	strTarget := TargetToREGEX(strTarget)
	arg_regx := "RegEx"
	strTarget := SubStr(strTarget,7)
}

/* 必須ウィンドウスタイルの扱い */
if( A_Args.Length >= 3 ){
	iAllowStyle := A_Args[3]
}
if( iAllowStyle = 0 || StrUpper(iAllowStyle) = "DEF" || StrUpper(iAllowStyle) = "DEFAULT" ){
	iAllowStyle := WS_VISIBLE
}else{
	iAllowStyle := StyleNumber(iAllowStyle,WS_VISIBLE)
}

/* 除外ウィンドウスタイルの扱い */
if( A_Args.Length >= 4 ){
	iDenyStyle := A_Args[4]
}
if( iDenyStyle = 0xFFFFFFFF || StrUpper(iDenyStyle) = "DEF" || StrUpper(iDenyStyle) = "DEFAULT" ){
	iDenyStyle := WS_DISABLED
}else{
	iDenyStyle := StyleNumber(iDenyStyle,WS_DISABLED)
}

/* エラーダイアログの扱い */
if( A_Args.Length >= 5 ){
	iDisp := A_Args[5]
}
if( iDisp = 1 || StrUpper(iDisp) = "ON" || StrUpper(iDisp) = "DISP" ){
	iDisp := 1
}else{
	iDisp := 0
}

/* デバッグモードの指定 */
if( A_Args.Length >= 6 ){
	iDebug := A_Args[6]
}
if( iDebug = 1 || StrUpper(iDebug) = "ON" || StrUpper(iDebug) = "DEBUG" ){
	iDebug := 1
	LogInit("TransColorWindow","AutoHotKeyV2_debug_log.txt")
}else{
	iDebug := 0
}

if( iDebug ){
	LogOutput( "PATH=" strTarget " REGX=" arg_regx " Allow=" StrHex8(iAllowStyle) " Deny=" StrHex8(iDenyStyle) )
}

/* タイトルマッチモードの設定(2=部分一致, "RegEx"=正規表現) */
SetTitleMatchMode(arg_regx)

/* フルパスに合致するウィンドウを列挙(非表示ウィンドウも対象としているため沢山ヒットする) */
listWin := WinGetList(strTarget)
idxWin := 0
findCount := 0

/* ウィンドウスタイルでマスク */
for(hWin in listWin){
	idxWin := idxWin + 1
	try{
		style := WinGetStyle("ahk_id " hWin)
		if( iDebug ){
			LogOutput("Style=" StrHex8(style) " Allow=" StrHex8(iAllowStyle) " Deny=" StrHex8(iDenyStyle))
		}
		if( TestStyle( style, iAllowStyle, iDenyStyle, hWin ) ){
			findCount++
			UniqueID := hWin
		}
	}
}

if( UniqueID = 0 ){
	if( iDisp ){
		MsgBox(strTarget " not found.")
	}
	if( iDebug ){
		LogOutput(strTarget " not found.")
	}
	ExitApp(1)
}

if( findCount != 1 ){
	if( iDisp ){
		MsgBox(strTarget " has multiple windows.")
	}
	if( iDebug ){
		LogOutput(strTarget " has multiple windows.")
	}
	ExitApp(1)
}

if( colorValue != "" && colorCode != "" ){
	try {
		WinSetTransColor( colorCode " " Number("0x" colorValue),"ahk_id " UniqueID )
		if( iDebug ){
			LogOutput("WinSetTransColor : hWnd=" UniqueID " Style=" StrHex8(Style) " Trans:" colorValue " / Color:" colorCode )
		}
		iRetCode := 0
	} catch TargetError as err {
		if( iDisp ){
			MsgBox( strTarget " not found[WinClose]" )
			iRetCode := 1
		}
		if( iDebug ){
			LogOutput( strTarget " not found[WinClose]" )
		}
	}
}else{
	try {
		WinSetTransparent( "OFF" ,"ahk_id " UniqueID )
		WinSetTransColor( "OFF" ,"ahk_id " UniqueID )
		if( iDebug ){
			LogOutput("WinSetTransparent/WinSetTransColor hWnd=" UniqueID " Style=" StrHex8(Style) " Trans:OFF / Color:OFF")
		}
		iRetCode := 0
	} catch TargetError as err {
		if( iDisp ){
			MsgBox( strTarget " not found[WinClose]" )
			iRetCode := 1
		}
		if( iDebug ){
			LogOutput( strTarget " not found[WinClose]" )
		}
	}
}

if( iDebug && iRetCode ){
	LogOutput( strTarget " retcode=" iRetCode )
}

ExitApp(iRetCode)

DispUsage(){
	msg := 'Usage:`n'
	msg := msg 'TransColorWindow.exe Color Target (...options) `n`n'
	msg := msg '[Color]`n'
	msg := msg '    0xAARRGGBB`n'
	msg := msg '    ex) 0xFFFF00FF (Alpha,Red,Green,Blue)`n'
	msg := msg '    none/off = Disable TransparentColor`n'
	msg := msg '[Target]`n'
	msg := msg '   "window-title / REGEX:match-title-str`n'
	msg := msg '    ahk_exe full-path`n'
	msg := msg '    ahk_class class-name..."`n'
	msg := msg '[AllowStyle]`n'
	msg := msg '    enabled WindowStyle Bits`n'
	msg := msg '    def/default = 0x10000000(WS_VISIBLE)`n'
	msg := msg '    ex) 0x16C70000`n'
	msg := msg '[DenyStyle]`n'
	msg := msg '    disabled WindowStyleBits`n'
	msg := msg '    def/default = 0x08000000(WS_DISABLED)`n'
	msg := msg '    ex) 0xE8000000`n'
	msg := msg '[DispError]`n'
	msg := msg '    1/on/disp = display ErrorDialog`n'
	msg := msg '[DebugMode]`n'
	msg := msg '    1/on/debug = Debug mode`n'
	MsgBox(msg)
	ExitApp(1)
}

