﻿;//-----------------------------------------------------------------------------------------------------------------
;// OpenWindow.ahk[UTF-8] (2026/02/10 14:00)
;// for AutoHotKey v2.0.21 or later
;//
;// http://coltpythonkingcobra.g1.xrea.com/pseudoTweetdeck/
;//                                             @PseudoTwDk
;//-----------------------------------------------------------------------------------------------------------------
;//
;
; 起動しようとしているウィンドウが既にあれば切替、無ければ起動します。
;    OpenWindow.exe Command Target (AllowStyle) (DenyStyle) (DispError) (Debug)
;       Command
;          起動するexeやbat、ショートカットとコマンドオプションを両端 "" 区切りで指定
;          ex) "explorer C:\WORK\TOOLS"
;       Target
;          ウィンドウ文字列/ahk_exe フルパス/ahk_class クラスID など両端""でAutoHotKey
;          独自のウィンドウ指定記述、現在アクティブなウィンドウを指定する場合は"A"
;          https://ahkscript.github.io/ja/docs/v2/misc/WinTitle.htm
;          対象ウィンドウのウィンドウスタイルは WS_VISIBLE ビットが有効で WS_DISABLED ビットが無効の
;          ものとしています。複数ウィンドウに該当する場合はエラー表示し処理しません。
;          マッチを正規表現で記述したい時は先頭に"REGEX:"を書きます
;          AutoHotKeyV2の正規表現文法は下記参照
;          https://ahkscript.github.io/ja/docs/v2/misc/RegEx-QuickRef.htm
;          ex) "ahk_exe C:\Program Files\Mozilla Firefox\firefox.exe"
;          ex) "ahk_exe C:\Program Files\Vivaldi\Application\vivaldi.exe ahk_class Chrome_WidgetWin_1"
;       AllowStyle
;          処理対象であるウィンドウスタイルのビット積(andをとってマスク自身になれば対象)の16進表記
;          省略された場合やdef/defaultと書かれた時は 0x10000000(WS_VISIBLE) が指定されたものとします
;          全てを対象外にしてしまう 0x00000000 は指定できません
;       DenyStyle
;          処理対象から除外するウィンドウスタイルのビット積(andをとって0になれば対象)の16進表記
;          省略された場合やdef/defaultと書かれた時は 0x08000000(WS_DISABLED) が指定されたものとします
;          全てを対象外にしてしまう 0xFFFFFFFF は指定できません
;       DispError
;          1/on/disp : 対象ウィンドウが見つからない時や複数ヒットした際のエラーダイアログを出す
;          [any] : 上記以外が指定されるか無指定ならエラー時にダイアログを出さない
;       DebugMode
;          1/on/debug : デバッグモードON
;          [any] : 上記以外が指定されるか無指定ならデバッグモードOFF
;
;       戻り値(exit-code)　※バッチファイルで本コマンド実行直後に %errorlevel% で参照出来ます
;          一意のウィンドウが該当した場合や無くて起動した場合はウィンドウの hWnd
;          起動出来なかった場合は 0
;

/* コマンドオプション数チェック */
if( A_Args.Length < 2 ){
	DispUsage()
}

/* 同時に複数実行を許可する */
#SingleInstance Off

/* 非表示になっているアプリも対象とする（仮想デスクトップ上では非表示扱い？） */
#WinActivateForce
DetectHiddenWindows True

#Include AhkCommon.ahk


/* 変数初期化 */
count := 0
iDisp := 0
iDebug := 0
iRetCode := 1
iAllowStyle := 0
iDenyStyle := 0xFFFFFFFF

strCommand := StrUpper(A_Args[1])
strTarget := StrReplace(A_Args[2], '"', "")
arg_regx := 2
if( StrUpper(SubStr(strTarget,1,6)) = "REGEX:" ){
	strTarget := TargetToREGEX(strTarget)
	arg_regx := "RegEx"
	strTarget := SubStr(strTarget,7)
}

/* 必須ウィンドウスタイルの扱い */
if( A_Args.Length >= 3 ){
	iAllowStyle := A_Args[3]
}
if( iAllowStyle = 0 || StrUpper(iAllowStyle) = "DEF" || StrUpper(iAllowStyle) = "DEFAULT" ){
	iAllowStyle := WS_VISIBLE
}else{
	iAllowStyle := StyleNumber(iAllowStyle,WS_VISIBLE)
}

/* 除外ウィンドウスタイルの扱い */
if( A_Args.Length >= 4 ){
	iDenyStyle := A_Args[4]
}
if( iDenyStyle = 0xFFFFFFFF || StrUpper(iDenyStyle) = "DEF" || StrUpper(iDenyStyle) = "DEFAULT" ){
	iDenyStyle := WS_DISABLED
}else{
	iDenyStyle := StyleNumber(iDenyStyle,WS_DISABLED)
}

/* エラーダイアログの扱い */
if( A_Args.Length >= 5 ){
	iDisp := A_Args[5]
}
if( iDisp = 1 || StrUpper(iDisp) = "ON" || StrUpper(iDisp) = "DISP" ){
	iDisp := 1
}else{
	iDisp := 0
}

/* デバッグモードの指定 */
if( A_Args.Length >= 6 ){
	iDebug := A_Args[6]
}
if( iDebug = 1 || StrUpper(iDebug) = "ON" || StrUpper(iDebug) = "DEBUG" ){
	iDebug := 1
	LogInit("OpenWindow","AutoHotKeyV2_debug_log.txt")
}else{
	iDebug := 0
}

if( iDebug ){
	LogOutput( "PATH=" strTarget " REGX=" arg_regx " Allow=" StrHex8(iAllowStyle) " Deny=" StrHex8(iDenyStyle) )
}

/* タイトルマッチモードの設定(2=部分一致, "RegEx"=正規表現) */
SetTitleMatchMode(arg_regx)

/* マッチするウィンドウの一覧を取得 */
listWin := WinGetList(strTarget)
idxWin := 0
findCount := 0
UniqueID := 0

/* ウィンドウスタイルでマスク */
for(hWin in listWin){
	idxWin := idxWin + 1
	try{
		style := WinGetStyle("ahk_id " hWin)
		if( iDebug ){
			LogOutput("Style=" StrHex8(style) " Allow=" StrHex8(iAllowStyle) " Deny=" StrHex8(iDenyStyle))
		}
		if( TestStyle( style, iAllowStyle, iDenyStyle, hWin ) ){
			findCount++
			;複数見つかった場合は最初にヒットしたハンドラを使う
			if( UniqueID = 0 ){
				UniqueID := hWin
			}
		}
	}
}

/* ウィンドウが存在しない場合起動する */
if( UniqueID = 0 ){
	try{
		Run( strCommand )
		UniqueID := WinWait( strTarget, ,5 )
		if( iDebug ){
			LogOutput("Run & WinWait OK : " strCommand " (" strTarget ")")
		}
	}catch{
		if( iDisp ){
			MsgBox("Run Error " strCommand " (" strTarget ")")
		}
		if( iDebug ){
			LogOutput("Run Error " strCommand " (" strTarget ")")
		}
	}
}

/* 見つかったウィンドウを前面にする */
if( UniqueID != 0 ){
	/* 最小化されていたら復元 */
	try{
		if( WinGetMinMax( "ahk_id" UniqueID ) = -1 ){
			WinRestore( "ahk_id" UniqueID )
			if( iDebug ){
				LogOutput("WinRestore : " UniqueID)
			}
		}
		WinActivate( "ahk_id" UniqueID )
		if( iDebug ){
			LogOutput("WinActivate : " UniqueID)
		}
	}catch{
		UniqueID := 0
	}
}

ExitApp(UniqueID)

DispUsage(){
	msg := 'Usage:`n'
	msg := msg 'OpenWindow.exe Command Target (...options) `n`n'
	msg := msg '[Command]`n'
	msg := msg '    "exe/bat/lnk option"`n'
	msg := msg '    ex) "explorer.exe C:\TEMP"`n'
	msg := msg '[Target]`n'
	msg := msg '   "window-title / REGEX:match-title-str`n'
	msg := msg '    ahk_exe full-path`n'
	msg := msg '    ahk_class class-name..."`n'
	msg := msg '[AllowStyle]`n'
	msg := msg '    enabled WindowStyle Bits`n'
	msg := msg '    def/default = 0x10000000(WS_VISIBLE)`n'
	msg := msg '    ex) 0x16C70000`n'
	msg := msg '[DenyStyle]`n'
	msg := msg '    disabled WindowStyleBits`n'
	msg := msg '    def/default = 0x08000000(WS_DISABLED)`n'
	msg := msg '    ex) 0xE8000000`n'
	msg := msg '[DispError]`n'
	msg := msg '    1/on/disp = display ErrorDialog`n'
	msg := msg '[DebugMode]`n'
	msg := msg '    1/on/debug = Debug mode`n'
	MsgBox(msg)
	ExitApp(1)
}

