﻿;//-----------------------------------------------------------------------------------------------------------------
;// MoveTopWindow.ahk[UTF-8] (2026/02/10 14:00)
;// for AutoHotKey v2.0.21 or later
;//
;// http://coltpythonkingcobra.g1.xrea.com/pseudoTweetdeck/
;//                                             @PseudoTwDk
;//-----------------------------------------------------------------------------------------------------------------
;//
;
; 指定のウィンドウのウィンドウスタイルを変更(生成時にしか変更出来ないものや無効となるものもあります)
;    MoveTopWindow.exe TopBottom Target (AllowStyle) (DenyStyle) (DispError) (Debug)
;       TopBottom
;          1/top : 最前面に変更してアクティブにする(アクティブにせず最前面はどうやら動作しない)
;          2/btm/bottom : ウィンドウを他のウィンドウよりも背面に変更する
;       Target
;          ウィンドウ文字列/ahk_exe フルパス/ahk_class クラスID など両端""でAutoHotKey
;          独自のウィンドウ指定記述、現在アクティブなウィンドウを指定する場合は"A"
;          https://ahkscript.github.io/ja/docs/v2/misc/WinTitle.htm
;          対象ウィンドウのウィンドウスタイルは WS_VISIBLE ビットが有効で WS_DISABLED ビットが無効の
;          ものとしています。複数ウィンドウに該当する場合はエラー表示し処理しません。
;          マッチを正規表現で記述したい時は先頭に"REGEX:"を書きます
;          AutoHotKeyV2の正規表現文法は下記参照
;          https://ahkscript.github.io/ja/docs/v2/misc/RegEx-QuickRef.htm
;          ex) "ahk_exe C:\Program Files\Mozilla Firefox\firefox.exe"
;          ex) "ahk_exe C:\Program Files\Vivaldi\Application\vivaldi.exe ahk_class Chrome_WidgetWin_1"
;       AllowStyle
;          処理対象であるウィンドウスタイルのビット積(andをとってマスク自身になれば対象)の16進表記
;          省略された場合やdef/defaultと書かれた時は 0x10000000(WS_VISIBLE) が指定されたものとします
;          全てを対象外にしてしまう 0x00000000 は指定できません
;       DenyStyle
;          処理対象から除外するウィンドウスタイルのビット積(andをとって0になれば対象)の16進表記
;          省略された場合やdef/defaultと書かれた時は 0x08000000(WS_DISABLED) が指定されたものとします
;          全てを対象外にしてしまう 0xFFFFFFFF は指定できません
;       DispError
;          1/on/disp : 対象ウィンドウが見つからない時や複数ヒットした際のエラーダイアログを出す
;          [any] : 上記以外が指定されるか無指定ならエラー時にダイアログを出さない
;       DebugMode
;          1/on/debug : デバッグモードON
;          [any] : 上記以外が指定されるか無指定ならデバッグモードOFF
;

/* 同時に複数実行を許可する */
#SingleInstance Off

#WinActivateForce
/* 非表示のウィンドウも捜索対象としておかないと仮想デスクトップで動作しない */
DetectHiddenWindows True

#Include AhkCommon.ahk


/* コマンドオプション数チェック */
if( A_Args.Length < 2 ){
	DispUsage()
}

/* モードとウィンドウ指定 */
iMode := A_Args[1]
strTarget := A_Args[2]
arg_regx := 2
if( StrUpper(SubStr(strTarget,1,6)) = "REGEX:" ){
	strTarget := TargetToREGEX(strTarget)
	arg_regx := "RegEx"
	strTarget := SubStr(strTarget,7)
}
UniqueID := 0
iDisp := 0
iDebug := 0
iAllowStyle := 0
iDenyStyle := 0xFFFFFFFF

if( iMode = 1 || StrUpper(iMode) = "TOP" ){
	iMode := 1
}else if( iMode = 2 || StrUpper(iMode) = "BTM" || StrUpper(iMode) = "BOTTOM" ){
	iMode := 2
}else{
	DispUsage()
}

/* 必須ウィンドウスタイルの扱い */
if( A_Args.Length >= 3 ){
	iAllowStyle := A_Args[3]
}
if( iAllowStyle = 0 || StrUpper(iAllowStyle) = "DEF" || StrUpper(iAllowStyle) = "DEFAULT" ){
	iAllowStyle := WS_VISIBLE
}else{
	iAllowStyle := StyleNumber(iAllowStyle,WS_VISIBLE)
}

/* 除外ウィンドウスタイルの扱い */
if( A_Args.Length >= 4 ){
	iDenyStyle := A_Args[4]
}
if( iDenyStyle = 0xFFFFFFFF || StrUpper(iDenyStyle) = "DEF" || StrUpper(iDenyStyle) = "DEFAULT" ){
	iDenyStyle := WS_DISABLED
}else{
	iDenyStyle := StyleNumber(iDenyStyle,WS_DISABLED)
}


/* エラーダイアログの扱い */
if( A_Args.Length >= 5 ){
	iDisp := A_Args[5]
}
if( iDisp = 1 || StrUpper(iDisp) = "ON" || StrUpper(iDisp) = "DISP" ){
	iDisp := 1
}else{
	iDisp := 0
}

/* デバッグモードの指定 */
if( A_Args.Length >= 6 ){
	iDebug := A_Args[6]
}
if( iDebug = 1 || StrUpper(iDebug) = "ON" || StrUpper(iDebug) = "DEBUG" ){
	iDebug := 1
	LogInit("MoveTopWindow","AutoHotKeyV2_debug_log.txt")
}else{
	iDebug := 0
}

if( iDebug ){
	LogOutput( "PATH=" strTarget " REGX=" arg_regx " Allow=" StrHex8(iAllowStyle) " Deny=" StrHex8(iDenyStyle) )
}

/* タイトルマッチモードの設定(2=部分一致, "RegEx"=正規表現) */
SetTitleMatchMode(arg_regx)

/* フルパスに合致するウィンドウを列挙(非表示ウィンドウも対象としているため沢山ヒットする) */
listWin := WinGetList(strTarget)
idxWin := 0
findCount := 0

/* ウィンドウスタイルでマスク */
for(hWin in listWin){
	idxWin := idxWin + 1
	try{
		style := WinGetStyle("ahk_id " hWin)
		if( iDebug ){
			LogOutput("Style=" StrHex8(style) " Allow=" StrHex8(iAllowStyle) " Deny=" StrHex8(iDenyStyle))
		}
		if( TestStyle( style, iAllowStyle, iDenyStyle, hWin ) ){
			findCount++
			UniqueID := hWin
		}
	}
}

if( UniqueID = 0 ){
	if( iDisp ){
		MsgBox(strTarget " not found.")
	}
	if( iDebug ){
		LogOutput(strTarget " not found.")
	}
	ExitApp(1)
}

if( findCount != 1 ){
	if( iDisp ){
		MsgBox(strTarget " has multiple windows.")
	}
	if( iDebug ){
		LogOutput(strTarget " has multiple windows.")
	}
	ExitApp(1)
}


try{
	if( iMode = 1 ){
		WinMoveTop("ahk_id " UniqueID)
		WinActivate("ahk_id " UniqueID)
		if( iDebug ){
			LogOutput("WinMoveTop & WinActivate : " UniqueID)
		}
	}else if( iMode = 2 ){
		WinMoveBottom("ahk_id " UniqueID)
		if( iDebug ){
			LogOutput("WinMoveBottom : " UniqueID)
		}
	}
}catch TargetError as err{
	if( iDisp ){
		if( iMode = 1 ){
			MsgBox("WinMoveTop Target Error.")
		}else if( iMode = 2 ){
			MsgBox("WinMoveBottom Target Error.")
		}
	}
	if( iDebug ){
		if( iMode = 1 ){
			LogOutput("WinMoveTop Target Error.")
		}else if( iMode = 2 ){
			LogOutput("WinMoveBottom Target Error.")
		}
	}
	ExitApp(1)
}catch OSError as osr{
	if( iDisp ){
		if( iMode = 1 ){
			MsgBox("WinMoveTop OS Error.")
		}else if( iMode = 2 ){
			MsgBox("WinMoveBottom OS Error.")
		}
	}
	if( iDebug ){
		if( iMode = 1 ){
			LogOutput("WinMoveTop OS Error.")
		}else if( iMode = 2 ){
			LogOutput("WinMoveBottom OS Error.")
		}
	}
	ExitApp(1)
}

ExitApp(0)




DispUsage(){
	msg := 'Usage:`n'
	msg := msg 'MoveTopWindow.exe TopBottom Target (...options)`n`n'
	msg := msg '[TopBottom]`n'
	msg := msg '    1/top = Set Window on Top`n'
	msg := msg '    2/btm/bottom = Set Window on Bottom`n'
	msg := msg '[Target]`n'
	msg := msg '   "window-title / REGEX:match-title-str`n'
	msg := msg '    ahk_exe full-path`n'
	msg := msg '    ahk_class class-name..."`n'
	msg := msg '[AllowStyle]`n'
	msg := msg '    enabled WindowStyle Bits`n'
	msg := msg '    def/default = 0x10000000(WS_VISIBLE)`n'
	msg := msg '    ex) 0x16C70000`n'
	msg := msg '[DenyStyle]`n'
	msg := msg '    disabled WindowStyleBits`n'
	msg := msg '    def/default = 0x08000000(WS_DISABLED)`n'
	msg := msg '    ex) 0xE8000000`n'
	msg := msg '[DispError]`n'
	msg := msg '    1/on/disp = display ErrorDialog`n'
	msg := msg '[DebugMode]`n'
	msg := msg '    1/on/debug = Debug mode`n'
	MsgBox(msg)
	ExitApp(1)
}


