﻿;//-----------------------------------------------------------------------------------------------------------------
;// CloseWindow.ahk[UTF-8] (2026/02/10 14:00)
;// for AutoHotKey v2.0.21 or later
;//
;// http://coltpythonkingcobra.g1.xrea.com/pseudoTweetdeck/
;//                                             @PseudoTwDk
;//-----------------------------------------------------------------------------------------------------------------
;//
;
; フルパスで指定されるウィンドウを終了します。
;    CloseWindow.exe Mode Target (AllowStyle) (DenyStyle) (DispError) (Debug)
;       Mode(C/K/CO/KO)
;          1/C/CLOSE     [複数ウィンドウ対象] ウィンドウを閉じる(アプリによっては確認ダイアログなどが出て終わらない),
;          2/K/KILL      [複数ウィンドウ対象] 強制終了(アプリによっては最終状態が保存されない)
;          3/CO/CLOSEONE [単一ウィンドウ対象] ウィンドウを閉じる(アプリによっては確認ダイアログなどが出て終わらない),
;          4/KO/KILLONE  [単一ウィンドウ対象] 強制終了(アプリによっては最終状態が保存されない)
;       Target
;          閉じるウィンドウのウィンドウ指定、両端は""で囲う
;          タイトル＋パスでマッチしたい時は "タイトル ahk_exe フルパス"指定
;          パスのみでマッチしたい時は "ahk_exe フルパス"指定
;          タイトルのみでマッチしたい時は "タイトル"指定
;          マッチを正規表現で記述したい時は先頭に"REGEX:"を書きます
;          AutoHotKeyV2の正規表現文法は下記参照
;          https://ahkscript.github.io/ja/docs/v2/misc/RegEx-QuickRef.htm
;       AllowStyle
;          処理対象であるウィンドウスタイルのビット積(andをとってマスク自身になれば対象)の16進表記
;          省略された場合やdef/defaultと書かれた時は 0x10000000(WS_VISIBLE) が指定されたものとします
;          全てを対象外にしてしまう 0x00000000 は指定できません
;       DenyStyle
;          処理対象から除外するウィンドウスタイルのビット積(andをとって0になれば対象)の16進表記
;          省略された場合やdef/defaultと書かれた時は 0x08000000(WS_DISABLED) が指定されたものとします
;          全てを対象外にしてしまう 0xFFFFFFFF は指定できません
;       DispError
;          1/on/disp : 対象ウィンドウが見つからない時や複数ヒットした際のエラーダイアログを出す
;          [any] : 上記以外が指定されるか無指定ならエラー時にダイアログを出さない
;       DebugMode
;          1/on/debug : デバッグモードON
;          [any] : 上記以外が指定されるか無指定ならデバッグモードOFF
;
;

/* コマンドオプション数チェック */
if( A_Args.Length < 2 ){
	DispUsage()
}

/* 同時に複数実行を許可する */
#SingleInstance Off

/* 非表示になっているアプリも対象とする（仮想デスクトップ上では非表示扱い？） */
#WinActivateForce
DetectHiddenWindows True

#Include AhkCommon.ahk


/* 変数初期化 */
count := 0
iDisp := 0
iDebug := 0
iRetCode := 1
iAllowStyle := 0
iDenyStyle := 0xFFFFFFFF

mode := StrUpper(A_Args[1])
if( mode == "1" || mode == "CLOSE" ){
	mode := "C"
}
if( mode == "2" || mode == "KILL" ){
	mode := "K"
}
if( mode == "3" || mode == "CLOSEONE" ){
	mode := "CO"
}
if( mode == "4" || mode == "KILLONE" ){
	mode := "KO"
}
strTarget := StrReplace(A_Args[2], '"', "")
arg_regx := 2
if( StrUpper(SubStr(strTarget,1,6)) = "REGEX:" ){
	strTarget := TargetToREGEX(strTarget)
	arg_regx := "RegEx"
	strTarget := SubStr(strTarget,7)
}

/* 必須ウィンドウスタイルの扱い */
if( A_Args.Length >= 3 ){
	iAllowStyle := A_Args[3]
}
if( iAllowStyle = 0 || StrUpper(iAllowStyle) = "DEF" || StrUpper(iAllowStyle) = "DEFAULT" ){
	iAllowStyle := WS_VISIBLE
}else{
	iAllowStyle := StyleNumber(iAllowStyle,WS_VISIBLE)
}

/* 除外ウィンドウスタイルの扱い */
if( A_Args.Length >= 4 ){
	iDenyStyle := A_Args[4]
}
if( iDenyStyle = 0xFFFFFFFF || StrUpper(iDenyStyle) = "DEF" || StrUpper(iDenyStyle) = "DEFAULT" ){
	iDenyStyle := WS_DISABLED
}else{
	iDenyStyle := StyleNumber(iDenyStyle,WS_DISABLED)
}

/* エラーダイアログの扱い */
if( A_Args.Length >= 5 ){
	iDisp := A_Args[5]
}
if( iDisp = 1 || StrUpper(iDisp) = "ON" || StrUpper(iDisp) = "DISP" ){
	iDisp := 1
}else{
	iDisp := 0
}

/* デバッグモードの指定 */
if( A_Args.Length >= 6 ){
	iDebug := A_Args[6]
}
if( iDebug = 1 || StrUpper(iDebug) = "ON" || StrUpper(iDebug) = "DEBUG" ){
	iDebug := 1
	LogInit("CloseWindow","AutoHotKeyV2_debug_log.txt")
}else{
	iDebug := 0
}

if( iDebug ){
	LogOutput( "PATH=" strTarget " REGX=" arg_regx " Allow=" StrHex8(iAllowStyle) " Deny=" StrHex8(iDenyStyle) " mode=" mode )
}

/* タイトルマッチモードの設定(2=部分一致, "RegEx"=正規表現) */
SetTitleMatchMode(arg_regx)

/* フルパスに合致するウィンドウを列挙(非表示ウィンドウも対象としているため沢山ヒットする) */
listWin := WinGetList(strTarget)
idxWin := 0
findCount := 0
UniqueID := 0

/* ウィンドウスタイルでマスク */
for(hWin in listWin){
	idxWin := idxWin + 1
	style := 0
	try{
		style := WinGetStyle("ahk_id " hWin)
		if( iDebug ){
			LogOutput("Style=" StrHex8(style) " Allow=" StrHex8(iAllowStyle) " Deny=" StrHex8(iDenyStyle))
		}
	}
	if( TestStyle( style, iAllowStyle, iDenyStyle, hWin ) ){
		findCount++
		UniqueID := hWin

		/* 複数ウィンドウが対象の場合 */
		if( mode == "C" ){
			try{
				WinClose( "ahk_id " UniqueID )
				if( iDebug ){
					LogOutput("WinClose : hWnd=" UniqueID " Style=" StrHex8(Style) " mode=" mode )
				}
			}catch TargetError as err{
				if( iDisp ){
					MsgBox("WinClose Target Error.")
				}
				if( iDebug ){
					LogOutput("WinClose Target Error.")
				}
			}catch OSError as osr{
				if( iDisp ){
					MsgBox("WinClose OSError.")
				}
				if( iDebug ){
					LogOutput("WinClose OSError.")
				}
			}
		} else if( mode == "K" ){
			try{
				WinKill( "ahk_id " UniqueID )
				if( iDebug ){
					LogOutput("WinKill : hWnd=" UniqueID " Style=" StrHex8(Style) " mode=" mode )
				}
			}catch TargetError as err{
				if( iDisp ){
					MsgBox("WinClose Target Error.")
				}
				if( iDebug ){
					LogOutput("WinClose Target Error.")
				}
			}catch OSError as osr{
				if( iDisp ){
					MsgBox("WinClose OSError.")
				}
				if( iDebug ){
					LogOutput("WinClose OSError.")
				}
			}
		}
	}
}


if( mode == "CO" || mode == "KO" ){
	if( UniqueID = 0 ){
		if( iDisp ){
			MsgBox(strTarget " not found.")
		}
		if( iDebug ){
			LogOutput(strTarget " not found.")
		}
		ExitApp(1)
	}

	if( findCount != 1 ){
		if( iDisp ){
			MsgBox(strTarget " has multiple windows.")
		}
		if( iDebug ){
			LogOutput(strTarget " has multiple windows.")
		}
		ExitApp(1)
	}
}


/* 単一ウィンドウが対象の場合 */
if( mode == "CO" ){
	try{
		WinClose( "ahk_id " UniqueID )
		if( iDebug ){
			LogOutput("WinClose : hWnd=" UniqueID " Style=" StrHex8(Style) " mode=" mode )
		}
	}catch TargetError as err{
		if( iDisp ){
			MsgBox("WinClose Target Error.")
		}
		if( iDebug ){
			LogOutput("WinClose Target Error.")
		}
		ExitApp(1)
	}catch OSError as osr{
		if( iDisp ){
			MsgBox("WinClose OSError.")
		}
		if( iDebug ){
			LogOutput("WinClose OSError.")
		}
		ExitApp(1)
	}
} else if( mode == "KO" ){
	if( iDebug ){
		LogOutput("hWnd=" UniqueID " Style=" StrHex8(Style) " mode=" mode )
	}
	try{
		WinKill( "ahk_id " UniqueID )
	}catch TargetError as err{
		if( iDisp ){
			MsgBox("WinKill Target Error.")
		}
		if( iDebug ){
			LogOutput("WinKill Target Error.")
		}
		ExitApp(1)
	}catch OSError as osr{
		if( iDisp ){
			MsgBox("WinKill OSError.")
		}
		if( iDebug ){
			LogOutput("WinKill OSError.")
		}
		ExitApp(1)
	}
}

ExitApp(0)

DispUsage(){
	msg := 'Usage:`n'
	msg := msg 'CloseWindow.exe Mode Target (...options) `n`n'
	msg := msg '[Mode]`n'
	msg := msg '    1/C/CLOSE = [multi file] close mode,`n'
	msg := msg '    2/K/KILL = [multi file] kill mode,`n'
	msg := msg '    3/CO/CLOSEONE = [one file] close mode,`n'
	msg := msg '    4/KO/KILLONE = [one file] kill mode,`n'
	msg := msg '[Target]`n'
	msg := msg '   "window-title / REGEX:match-title-str`n'
	msg := msg '    ahk_exe full-path`n'
	msg := msg '    ahk_class class-name..."`n'
	msg := msg '[AllowStyle]`n'
	msg := msg '    enabled WindowStyle Bits`n'
	msg := msg '    def/default = 0x10000000(WS_VISIBLE)`n'
	msg := msg '    ex) 0x16C70000`n'
	msg := msg '[DenyStyle]`n'
	msg := msg '    disabled WindowStyleBits`n'
	msg := msg '    def/default = 0x08000000(WS_DISABLED)`n'
	msg := msg '    ex) 0xE8000000`n'
	msg := msg '[DispError]`n'
	msg := msg '    1/on/disp = display ErrorDialog`n'
	msg := msg '[DebugMode]`n'
	msg := msg '    1/on/debug = Debug mode`n'
	MsgBox(msg)
	ExitApp(1)
}

