﻿;//-----------------------------------------------------------------------------------------------------------------
;// AhkCommon.ahk[UTF-8] (2026/02/22 02:30)
;// for AutoHotKey v2.0.21 or later
;//
;// http://coltpythonkingcobra.g1.xrea.com/pseudoTweetdeck/
;//                                             @PseudoTwDk
;//-----------------------------------------------------------------------------------------------------------------
;//
;// 共通関数定義
;// 使用ソースの冒頭に「#Include AhkCommon.ahk」でインクルードして使う
;//
;// 定義関数一覧
;//   StrLenA                     マルチバイト対応のAscii文字数カウント
;//   SubStrA                     マルチバイト対応のAscii文字数切り出し
;//   StrRepeat                   同じ文字の繰り返し
;//   StrNum                      ゼロパディング整形
;//   StrHex                      ゼロパディングHEX整形
;//   StrSpace                    スペースパディング整形
;//   StrSpaceA                   スペースパディング整形(Ascii文字列換算)
;//     StrNum2                     2桁ゼロパディング整形
;//     StrNum3                     3桁ゼロパディング整形
;//     StrNum4                     4桁ゼロパディング整形
;//     StrNum5                     5桁ゼロパディング整形
;//     StrNum6                     6桁ゼロパディング整形
;//     StrNum7                     7桁ゼロパディング整形
;//     StrNum8                     8桁ゼロパディング整形
;//     StrHex2                     2桁ゼロパディングHEX整形
;//     StrHex4                     4桁ゼロパディングHEX整形
;//     StrHex8                     8桁ゼロパディングHEX整形
;//     StrHex16                    16桁ゼロパディングHEX整形
;//     StrSpaceRight               スペースパディング右寄せ整形
;//     StrSpaceLeft                スペースパディング左寄せ整形
;//     StrSpaceRightA              スペースパディング右寄せ整形(Ascii文字列換算)
;//     StrSpaceLeftA               スペースパディング左寄せ整形(Ascii文字列換算)
;//   ByteToBuff                  バイト列からバッファにバイナリの格納
;//   BuffToByte                  バッファからバイト列にバイナリの取得
;//   ByteToString                バイト列を文字列化
;//   StringToByte                文字列をバイト列化
;//   ByteCompare                 バイト列同士を比較
;//   BuffCompare                 バッファ同士を比較
;//
;//   TargetParse                 ウィンドウ検索文字列を分析する
;//     PathToREGEX                 未エスケープのパス文字列をRegEx対応の文字列に変換
;//     REGEXToPath                 エスケープ済みのパス文字列を通常のパス文字列に変換
;//   ConvertREGEX                ウィンドウ検索文字列を分析し未エスケープのパスとエスケープ済みのパスを相互変換
;//     TargetToREGEX               ウィンドウ検索文字列を分析し未エスケープのパスを変換
;//     REGEXToTarget               ウィンドウ検索文字列を分析しエスケープ済みのパスを逆変換
;//
;//   ShowCustomToolTip           色指定やセンタリングが可能なカスタムツールチップ表示
;//   FadeoutCustomToolTip        カスタムツールチップのフェードアウト消去
;//   DestroyCustomToolTip        カスタムツールチップの消去
;//   OnEvent_CustomToolTip       カスタムツールチップのイベント処理
;//     OnClick_CustomToolTip       左クリックイベントのコールバック
;//     OnDblClick_CustomToolTip    左ダブルクリックイベントのコールバック
;//     OnLClick_CustomToolTip      左クリック共通処理
;//     OnRClick_CustomToolTip      右クリックイベントのコールバック
;//     OnEsc_CustomToolTip         ESCボタン押下イベントのコールバック
;//
;//   LogInit                     ログ出力の初期処理
;//   LogOutput                   ログ出力
;//     IsLogEnable                 ログが初期化されているかどうかを返す
;//
;//   StyleNumber                 ウィンドウスタイル定義名から定義値に変換
;//   TestStyle                   ウィンドウスタイルの必須ビット除外ビットにマッチするか判定
;//   GetWindowStyleStr           ウィンドウスタイル定義名・定義値のUsage用文字列を返す
;//   GetWindowStyleExStr         拡張ウィンドウスタイル定義名・定義値のUsage用文字列を返す
;//
;//   CheckAudioChange            音声が再生中かどうか返す
;//     OnAudioChange             音声の再生/停止状態が変化したときに呼ばれる
;//
;//   SystemCursor                マウスカーソルの変更
;//
;//   SyncPause                   停止再開を切り替え、他の常駐アプリに通知する
;//   SyncStart                   即時開始を実行し、他の常駐アプリに通知する
;//
;//
;// 定義グローバル変数・定数一覧
;//   AHK_COMMON_LOG_INDEX        共通関数自身のデバッグをする時のLogIndex
;//   gA_CustomToolTipGui         カスタムツールチップオブジェクト配列
;//   gA_CustomToolTipOpt         カスタムツールチップ設定配列
;//   gA_LogOutput                ログ出力設定配列
;//   WS_xxx                      ウィンドウスタイル定義類
;//   WS_EX_xxx                   拡張ウィンドウスタイル定義類
;//   gi_audioPlaying             音声が再生中かどうかのチェック時最終状態
;//   MSG_xxx                     プロセス間通信のメッセージ番号類
;//   APP_xxx                     プロセス固有の番号類
;//   gObj_SyncPauseAppList       プロセス間通信で一時停止と再開を配信する一覧
;//   gObj_SyncStartAppList       プロセス間通信で即時実行を配信する一覧
;//
;//
;//-----------------------------------------------------------------------------------------------------------------










;//共通関数自体のデバッグをするときは下記のグローバル変数をテストするソース側で1以上に設定した上でLogInitを実行すること
global AHK_COMMON_LOG_INDEX := 0

;//-----------------------------------------------------------------------------------------------------------------
;// StrLenA マルチバイトのUnicodeを考慮して文字列の長さを返す関数
;//  str :        対象の文字列
;//  戻り値 :     Ascii換算の文字数
;//-----------------------------------------------------------------------------------------------------------------
StrLenA( str ){
	
	;//Unicodeとして文字数が扱われるのでAscii換算
	return Number(StrPut(str, "cp0") - 1)
}

;//-----------------------------------------------------------------------------------------------------------------
;// SubStrA マルチバイトのUnicodeを考慮して指定文字数で文字列を切り出す関数
;//  str :        切り出し元の文字列
;//  pos :        切り出し位置(1～ または後方から切り出す場合は -1以下) ※0は指定できないため空文字を返す
;//  len :        切り出す長さをAscii換算で指定(1～ または末尾を削る場合は -1以下) ※0指定時は開始位置以降全てを返す
;//  戻り値 :     切り出された文字列
;//-----------------------------------------------------------------------------------------------------------------
SubStrA( str, pos, len:=0 ){

	out := ""
	tmp := ""
	idx := 0
	count := 0

	;//開始位置が0や長さ以上の場合は空文字を返す
	if( pos = 0 || pos > StrLenA(str) ){
		return out
	}
	;//開始位置が負数で左端を越える場合は先頭から全て
	if( -1 * pos > StrLenA(str) ){
		pos := 1
		len := 0
	}
	;//長さが負数の場合はその文字数末尾から削る
	if( len < 0 ){
		len := StrLenA(str) + len - pos + 1
	}
	;//長さが0の時は開始位置以降を返す
	if( len = 0 ){
		len := StrLenA(str)
	}
	
	if( pos > 0 ){
		;//順方向
		While( idx < pos ){
			tmp := SubStr( str, idx, 1 )
			if( StrLenA(tmp) = 2 ){
				pos--
				if( idx = pos ){
					out := "?" out
					len--
					pos++
				}
			}
			idx++
		}
		idx := 0
		While( count < len ){
			tmp := SubStr( str, pos + idx, 1 )
			if( StrLenA(tmp) = 2 ){
				count++
				;//全角文字が足される事で長さを超える時
				if( count = len ){
					out := out "?"
				}else{
					out := out tmp
				}
			}else{
				out := out tmp
			}
			idx++
			count++
		}
	}else{
		;//逆方向の時は位置が負数指定されているので絶対値ぶん末尾から取得
		len := -1 * pos
		While( count < len ){
			tmp := SubStr( str, -1 )
			str := SubStr( str, 1, StrLen(str) - 1 )
			if( StrLenA(tmp) = 2 ){
				count++
				;//全角文字が足される事で長さを超える時
				if( count = len ){
					out := "?" out
				}else{
					out := tmp out
				}
			}else{
				out := tmp out
			}
			count++
		}
	}
	return out
}


/* SubStrA 使用例

SubStrA(title, 7, 10) ;始点と長さを指定
SubStrA(title, 7)     ;長さを省略
SubStrA(title, -7)    ;始点を負数にして逆から切り取り
SubStrA(title, 3, -9) ;長さを負数にして後ろから削除
SubStrA(title, 3, 11) ;終端が全角で分断されて ? になる
SubStrA(title, 2, 11) ;先頭が全角で分断されて ? になる
SubStrA(title, 2, 10) ;先頭と終端が全角で両端分断され ? になる
SubStrA(title, 0, 11) ;位置ゼロは空文字を返す

*/






;//-----------------------------------------------------------------------------------------------------------------
;// StrRepeat  同じ文字の繰り返し文字列
;//  str :        繰り返す文字
;//  len :        桁数
;//  戻り値 :     繰り返し文字列
;//-----------------------------------------------------------------------------------------------------------------
StrRepeat( str, len ){
	strTemp := ""
	idx := len
	While( idx ){
		strTemp := strTemp str
		idx--
	}
	return strTemp
}


;//-----------------------------------------------------------------------------------------------------------------
;// StrNum  ゼロパディング整形
;//  num :        数値
;//  len :        桁数
;//  戻り値 :     パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrNum( num, len ){
	if( IsInteger("" num) ){
		return SubStr(StrRepeat("0",len) String(num), -1 * len)
	}else{
		return StrRepeat("?",len)
	}
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrHex  ゼロパディングHEX整形
;//  num :        数値
;//  len :        桁数
;//  pre :        HEXプリフィクス "0x" を先頭につける=1, つけない=0
;//  戻り値 :     パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrHex( num, len, pre:=0 ){
	strPre := "0x"
	if( pre = 0 ){
		strPre := ""
	}
	if( IsInteger("" num) ){
		return strPre SubStr(StrRepeat("0",len) StrUpper(Format("{1:X}",num)), -1 * len)
	}else{
		return strPre StrRepeat("?",len)
	}
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrSpace  スペースパディング整形
;//  str :        文字列
;//  len :        桁数
;//  pos :        右寄せ=0, 左寄せ=1
;//  戻り値 :     パディングされた文字列
;//-----------------------------------------------------------------------------------------------------------------
StrSpace( str, len, pos ){
	strSpc := StrRepeat(" ",len)
	strOut := ""
	switch( pos ){
	case 0:
		strOut := strSpc str
		strOut := SubStr(strOut, -1 * len)
	case 1:
		strOut := str strSpc
		strOut := SubStr(strOut, 1, len)
	}
	return strOut
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrSpaceA  スペースパディング整形(Ascii文字列換算)
;//  str :        文字列
;//  lenA:        桁数(Ascii文字列換算)
;//  pos :        右寄せ=0, 左寄せ=1
;//  戻り値 :     パディングされた文字列
;//-----------------------------------------------------------------------------------------------------------------
StrSpaceA( str, lenA, pos ){
	strSpc := StrRepeat(" ",lenA)
	strOut := ""
	switch( pos ){
	case 0:
		strOut := strSpc str
		strOut := SubStrA(strOut, -1 * lenA)
	case 1:
		strOut := str strSpc
		strOut := SubStrA(strOut, 1, lenA)
	}
	return strOut
}


;//-----------------------------------------------------------------------------------------------------------------
;// StrNum2  2桁ゼロパディング整形
;//  num :        数値
;//  戻り値 :     00 パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrNum2( num ){
	return StrNum( num, 2 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// Str3Num  3桁ゼロパディング整形
;//  num :        数値
;//  戻り値 :     000 パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrNum3( num ){
	return StrNum( num, 3 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrNum4  4桁ゼロパディング整形
;//  num :        数値
;//  戻り値 :     0000 パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrNum4( num ){
	return StrNum( num, 4 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrNum5  5桁ゼロパディング整形
;//  num :        数値
;//  戻り値 :     00000 パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrNum5( num ){
	return StrNum( num, 5 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrNum6  6桁ゼロパディング整形
;//  num :        数値
;//  戻り値 :     000000 パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrNum6( num ){
	return StrNum( num, 6 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrNum7  7桁ゼロパディング整形
;//  num :        数値
;//  戻り値 :     0000000 パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrNum7( num ){
	return StrNum( num, 7 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrNum8  8桁ゼロパディング整形
;//  num :        数値
;//  戻り値 :     00000000 パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrNum8( num ){
	return StrNum( num, 8 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrHex2  2桁ゼロパディングHEX整形
;//  num :        数値
;//  戻り値 :     0x00 パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrHex2( num ){
	return StrHex( num, 2, 1 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrHex4  4桁ゼロパディングHEX整形
;//  num :        数値
;//  戻り値 :     0x0000 パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrHex4( num ){
	return StrHex( num, 4, 1 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrHex8  8桁ゼロパディングHEX整形
;//  num :        数値
;//  戻り値 :     0x00000000 パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrHex8( num ){
	return StrHex( num, 8, 1 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrHex16  16桁ゼロパディングHEX整形
;//  num :        数値
;//  戻り値 :     0x0000000000000000 パディングされた文字列/引数が数値でない場合は?文字
;//-----------------------------------------------------------------------------------------------------------------
StrHex16( num ){
	return StrHex( num, 16, 1 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrSpaceRight  スペースパディング右寄せ整形
;//  str :        文字列
;//  len :        桁数
;//  戻り値 :     パディングされた文字列
;//-----------------------------------------------------------------------------------------------------------------
StrSpaceRight( str, len ){
	return StrSpace( str, len, 0 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrSpaceLeft  スペースパディング左寄せ整形
;//  str :        文字列
;//  len :        桁数
;//  戻り値 :     パディングされた文字列
;//-----------------------------------------------------------------------------------------------------------------
StrSpaceLeft( str, len ){
	return StrSpace( str, len, 1 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrSpaceRightA  スペースパディング右寄せ整形(Ascii文字列換算)
;//  str :        文字列
;//  lenA:        桁数(Ascii文字列換算)
;//  戻り値 :     パディングされた文字列
;//-----------------------------------------------------------------------------------------------------------------
StrSpaceRightA( str, lenA ){
	return StrSpaceA( str, lenA, 0 )
}

;//-----------------------------------------------------------------------------------------------------------------
;// StrSpaceLeftA  スペースパディング左寄せ整形(Ascii文字列換算)
;//  str :        文字列
;//  lenA:        桁数(Ascii文字列換算)
;//  戻り値 :     パディングされた文字列
;//-----------------------------------------------------------------------------------------------------------------
StrSpaceLeftA( str, lenA ){
	return StrSpaceA( str, lenA, 1 )
}









;//-----------------------------------------------------------------------------------------------------------------
;// ByteToBuff  バイト列からバッファにバイナリの格納
;//  &buff   :    書き出し先のバッファ参照（先に := Buffer(len)で確保しておくこと）
;//  byteAry :    バイト配列 ([0xFF,0x00,...]形式)
;//  len     :    バイト長
;//  offset  :    書き出しオフセット(0～)
;//  index   :    読み込み先頭インデックス(1～)
;//  戻り値 :     成功時バイト長, 失敗時0
;//-----------------------------------------------------------------------------------------------------------------
ByteToBuff( &buff, byteAry, len, offset:=0, index:=1 ){

	iLoop := 1
	if( buff.Size < len || len < 1 ){
		return 0
	}
	try{
		while( len >= iLoop ){
			NumPut( "UChar", byteAry[iLoop + index - 1], buff, iLoop - 1 + offset )
			if( AHK_COMMON_LOG_INDEX ){
				LogOutput( "ByteToBuff(offset=" iLoop - 1 + offset ",index=" iLoop + index - 1 ") "
				. "byteAry[" iLoop + index - 1 "] = " StrHex2(byteAry[iLoop + index - 1]), AHK_COMMON_LOG_INDEX )
			}
			iLoop++
		}
	}catch{
		return 0
	}
	return len
}

;//-----------------------------------------------------------------------------------------------------------------
;// BuffToByte  バッファからバイト列にバイナリの取得
;//  &byteAry:    取得データ格納先のバイト配列参照 ([0xFF,0x00,...]形式)（先に := Array()で生成しておくこと）
;//  buff    :    取得元のバッファ
;//  len     :    バイト長
;//  offset  :    読み込みオフセット(0～)
;//  index   :    書き出し先頭インデックス(1～)
;//  戻り値 :     成功時バイト長, 失敗時0
;//-----------------------------------------------------------------------------------------------------------------
BuffToByte( &byteAry, buff, len, offset:=0, index:=1 ){

	iLoop := 1
	if( buff.Size < len || len < 1 ){
		return 0
	}
	if( byteAry.Length < len ){
		byteAry.Length := len
	}
	try{
		while( len >= iLoop ){
			byteAry[iLoop + index - 1] := NumGet( buff, iLoop - 1 + offset, "UChar" )
			if( AHK_COMMON_LOG_INDEX ){
				LogOutput( "BuffToByte(offset=" offset ",index=" index ") "
				. "byteAry[" iLoop + index - 1 "] = " StrHex2(byteAry[iLoop + index - 1]), AHK_COMMON_LOG_INDEX )
			}
			iLoop++
		}
	}catch{
		return 0
	}
	return len
}

;//-----------------------------------------------------------------------------------------------------------------
;// ByteToString  バイト列を文字列化
;//  byteAry :    バイト配列 ([0xFF,0x00,...]形式)
;//  len     :    バイト長
;//  戻り値 :     連結した16進表記の文字列(FF00...形式), 失敗時""
;//-----------------------------------------------------------------------------------------------------------------
ByteToString( &byteAry, len ){

	iLoop := 1
	str := ""
	try{
		while( len >= iLoop ){
			str := str StrHex( byteAry[iLoop], 2, 0 )
			if( AHK_COMMON_LOG_INDEX ){
				LogOutput( "ByteToString str[" iLoop "]=" StrHex( byteAry[iLoop], 2, 0 ), AHK_COMMON_LOG_INDEX )
			}
			iLoop++
		}
	}catch{
		return ""
	}
	return str
}

;//-----------------------------------------------------------------------------------------------------------------
;// StringToByte  文字列をバイト列化
;//  &byteAry:    取得データ格納先のバイト配列参照 ([0xFF,0x00,...]形式)（先に := Array()で生成しておくこと）
;//  str     :    取得元の文字列 (FF00...形式)
;//  len     :    バイト長
;//  戻り値 :     成功時バイト長, 失敗時0
;//-----------------------------------------------------------------------------------------------------------------
StringToByte( &byteAry, str, len ){

	err := 0
	iLoop := 1
	if( byteAry.Length < len ){
		byteAry.Length := len
	}
	str := StrReplace(str, "0x", "")
	try{
		while( len >= iLoop ){
			tmp := SubStr( str, (iLoop - 1) * 2 + 1, 2 )
			if( IsXDigit("0x" tmp) ){
				byteAry[iLoop] := Number("0x" SubStr( str, (iLoop - 1) * 2 + 1, 2 ))
			}else{
				byteAry[iLoop] := "??"
				err := 1
			}
			if( AHK_COMMON_LOG_INDEX ){
				LogOutput( "StringToByte byteAry[" iLoop "]=" StrHex2(byteAry[iLoop]), AHK_COMMON_LOG_INDEX )
			}
			iLoop++
		}
	}catch{
		return 0
	}
	if( err ){
		return 0
	}else{
		return len
	}
}

;//-----------------------------------------------------------------------------------------------------------------
;// ByteCompare  バイト列同士を比較
;//  byteAry1:    バイト配列比較対象1
;//  index1  :    比較対象1の先頭インデックス(1～)
;//  byteAry2:    バイト配列比較対象2
;//  index2  :    比較対象2の先頭インデックス(1～)
;//  len     :    比較バイト長(1～)
;//  戻り値  :     一致時1, 不一致時0, パラメータ不正時-1
;//-----------------------------------------------------------------------------------------------------------------
ByteCompare( byteAry1, index1, byteAry2, index2, len ){

	if( len < 1 || index1 < 1 || index2 < 1 ){
		return -1
	}
	if( byteAry1.Length < index1 + len - 1 ){
		return -1
	}
	if( byteAry2.Length < index2 + len - 1 ){
		return -1
	}
	idx := 1
	while( idx <= len ){
		if( AHK_COMMON_LOG_INDEX ){
			LogOutput( "ByteCompare byteAry1[" idx "]=[" StrHex2(byteAry1[index1 + idx - 1]) "] "
			. "/ byteAry2[" idx "]=[" StrHex2(byteAry2[index2 + idx - 1]) "]", AHK_COMMON_LOG_INDEX )
		}
		if( byteAry1[index1 + idx - 1] != byteAry2[index2 + idx - 1] ){
			return 0
		}
		idx++
	}
	return 1
}

;//-----------------------------------------------------------------------------------------------------------------
;// BuffCompare  バッファ同士を比較
;//  buff1   :    バッファ比較対象1
;//  offset1 :    比較対象1の先頭オフセット(0～)
;//  buff2   :    バッファ比較対象2
;//  offset2 :    比較対象2の先頭オフセット(0～)
;//  len     :    比較バイト長(1～)
;//  戻り値  :     一致時1, 不一致時0, パラメータ不正時-1
;//-----------------------------------------------------------------------------------------------------------------
BuffCompare( buff1, offset1, buff2, offset2, len ){

	if( len < 1 || offset1 < 0 || offset2 < 0 ){
		return -1
	}
	if( buff1.Size < offset1 + len ){
		return -1
	}
	if( buff2.Size < offset2 + len ){
		return -1
	}
	
	byteAry1 := Array()
	byteAry1.Length := len
	iRet := BuffToByte( &byteAry1, buff1, len, offset1 )
	if( iRet != len ){
		return -1
	}
	if( AHK_COMMON_LOG_INDEX ){
		for(idx, tmp in byteAry1){
			LogOutput( "BuffCompare(byteAry1) idx=" idx " [" StrHex2(tmp) "]", AHK_COMMON_LOG_INDEX )
		}
	}
	
	byteAry2 := Array()
	byteAry2.Length := len
	iRet := BuffToByte( &byteAry2, buff2, len, offset2 )
	if( iRet != len ){
		return -1
	}
	
	if( AHK_COMMON_LOG_INDEX ){
		for(idx, tmp in byteAry2){
			LogOutput( "BuffCompare(byteAry2) idx=" idx " [" StrHex2(tmp) "]", AHK_COMMON_LOG_INDEX )
		}
	}
	
	idx := 1
	while( idx <= len ){
		if( AHK_COMMON_LOG_INDEX ){
			LogOutput( "BuffCompare byteAry1[" idx "]=[" StrHex2(byteAry1[idx]) "] "
			. "/ byteAry2[" idx "]=[" StrHex2(byteAry2[idx]) "]", AHK_COMMON_LOG_INDEX )
		}
		if( byteAry1[idx] != byteAry2[idx] ){
			return 0
		}
		idx++
	}
	return 1
}



/* 使用例

AHK_COMMON_LOG_INDEX := 1

try{
	FileDelete("AhkCommonTest_LOG.txt")
}
LogInit("Test","AhkCommonTest_LOG.txt",,0,,AHK_COMMON_LOG_INDEX,1000)
LogOutput( "■Start", AHK_COMMON_LOG_INDEX)

buf1 := Buffer(3)
buf2 := Buffer(4)
buf3 := Buffer(3)
buf4 := Buffer(4)
orgAry1 := [ 0xEF, 0xBB, 0xBF ]
orgAry2 := [ 0x00, 0xEF, 0xBB, 0xBF ]
newAry := Array()
trdAry := Array()

iRet := ByteToBuff( &buf1, orgAry1, 3 )
LogOutput( "■00. ByteToBuff ret=" iRet, AHK_COMMON_LOG_INDEX )
;iRet=3(3バイト変換正常)

iRet := BuffToByte( &newAry, buf1, 3 )
LogOutput( "■01. BuffToByte ret=" iRet, AHK_COMMON_LOG_INDEX )
;iRet=3(3バイト変換正常)

str := ByteToString( &newAry, 3 )
LogOutput( "■02. str=" str, AHK_COMMON_LOG_INDEX )
;str=EFBBBF

iRet := StringToByte( &trdAry, str, 3 )
LogOutput( "■03. BuffToByte ret=" iRet, AHK_COMMON_LOG_INDEX )
;iRet=3(3バイト変換正常)

for(idx, tmp in trdAry){
	LogOutput( "■04. idx=" idx " [" StrHex2(tmp) "]", AHK_COMMON_LOG_INDEX )
	;idx=1[0xEF], idx=2[0xBB], idx=3[0xBF]
}

iRet := ByteCompare( orgAry1, 1, trdAry, 1, 3 )
LogOutput( "■05. ByteCompare ret=" iRet, AHK_COMMON_LOG_INDEX )
;iRet=1(比較一致)

iRet := ByteToBuff( &buf2, orgAry2, 4, 0, 1 )
LogOutput( "■06.ByteToBuff(len=4,offset=0,index=1) ret=" iRet, AHK_COMMON_LOG_INDEX )
;iRet=4(4バイト変換正常)

iRet := BuffCompare( buf1, 0, buf2, 1, 3 )
LogOutput( "■07.BuffCompare(offset1=0,offset2=1,len=3) ret=" iRet, AHK_COMMON_LOG_INDEX )
;iRet=1(比較一致)

iRet := ByteToBuff( &buf3, orgAry2, 3, 0, 2 )
LogOutput( "■08.ByteToBuff(len=3,offset=0,index=2) ret=" iRet, AHK_COMMON_LOG_INDEX )
;iRet=3(3バイト変換正常)

iRet := BuffCompare( buf1, 0, buf3, 0, 3 )
LogOutput( "■09.BuffCompare(offset1=0,offset2=0,len=3) ret=" iRet, AHK_COMMON_LOG_INDEX )
;iRet=1(比較一致)

iRet := BuffCompare( buf2, 0, buf3, 0, 4 )
LogOutput( "■10.BuffCompare(offset1=0,offset2=0,len=4) ret=" iRet, AHK_COMMON_LOG_INDEX )
;iRet=-1(buf3が3バイトなのにlen=4)

iRet := ByteToBuff( &buf4, orgAry2, 3, 0, 1 )
LogOutput( "■11.ByteToBuff(len=3,offset=0,index=1) ret=" iRet, AHK_COMMON_LOG_INDEX )
;iRet=3(3バイト変換正常)

iRet := BuffCompare( buf2, 0, buf4, 0, 4 )
LogOutput( "■12.BuffCompare(offset1=0,offset2=0,len=4) ret=" iRet, AHK_COMMON_LOG_INDEX )
;iRet=0(buf4は3バイトしか書いてないので4バイト目不一致)

*/





;//-----------------------------------------------------------------------------------------------------------------
;// TargetParse  ウィンドウ検索文字列を分析する ※タイトル, EXE-Path, Class, ID に分離
;//  strTarget :   ウィンドウ検索文字列
;//  &objParse :   解析結果オブジェクト(連想配列)
;//  戻り値    :   正常終了=0, 失敗=-1
;//-----------------------------------------------------------------------------------------------------------------
TargetParse( strTarget, &objParse ){
	
	objParse := {}
	objParse.Title := ""
	objParse.Path := ""
	objParse.Class := ""
	objParse.ID := 0
	
	strTgt := strTarget
	
	if( AHK_COMMON_LOG_INDEX ){
		LogOutput( "TargetParse: [" strTarget "]", AHK_COMMON_LOG_INDEX )
	}
	
	pos1 := InStr(strTgt,"ahk_exe ")
	pos2 := InStr(strTgt,"ahk_class ")
	pos3 := InStr(strTgt,"ahk_id ")

	;if( AHK_COMMON_LOG_INDEX ){
	;	LogOutput( "pos1=" pos1 " pos2=" pos2 " pos3=" pos3, AHK_COMMON_LOG_INDEX )
	;}
	
	;//タイトルのみのとき
	if( pos1 = 0 && pos2 = 0 && pos3 = 0 ){
		objParse.Title := strTgt
		if( AHK_COMMON_LOG_INDEX ){
			LogOutput( "objParse: out1[Title=" objParse.Title "]", AHK_COMMON_LOG_INDEX )
		}
		return 0
	}

	;//ahk_exe, ahk_class, ahk_id は1回ずつしか出現を許さない
	if( pos1 ){
		tmpAry := StrSplit(strTgt,"ahk_exe ")
		if( tmpAry.Length > 2 ){
			if( AHK_COMMON_LOG_INDEX ){
				LogOutput( "ahk_exeが複数定義されているエラー" pos3, AHK_COMMON_LOG_INDEX )
			}
			return -1
		}
	}
	if( pos2 ){
		tmpAry := StrSplit(strTgt,"ahk_class ")
		if( tmpAry.Length > 2 ){
			if( AHK_COMMON_LOG_INDEX ){
				LogOutput( "ahk_classが複数定義されているエラー" pos3, AHK_COMMON_LOG_INDEX )
			}
			return -1
		}
	}
	if( pos3 ){
		tmpAry := StrSplit(strTgt,"ahk_id ")
		if( tmpAry.Length > 2 ){
			if( AHK_COMMON_LOG_INDEX ){
				LogOutput( "ahk_idが複数定義されているエラー" pos3, AHK_COMMON_LOG_INDEX )
			}
			return -1
		}
	}
	
	tmpAry := StrSplit(strTgt,["ahk_exe ","ahk_class ","ahk_id "])
	;//タイトルは左端固定
	objParse.Title := tmpAry[1]
	
	if( tmpAry.Length > 4 ){
		if( AHK_COMMON_LOG_INDEX ){
			LogOutput( "5つ以上に分割されてしまったエラー", AHK_COMMON_LOG_INDEX )
		}
		return -1
	}
	
	if( tmpAry.Length = 4 ){
		if( pos1 > pos2 && pos1 > pos3 ){
			objParse.Path := tmpAry[4]
			if( pos2 > pos3 ){
				objParse.Class := tmpAry[3]
				objParse.ID := tmpAry[2]
			}else{
				objParse.Class := tmpAry[2]
				objParse.ID := tmpAry[3]
			}
		}else if( pos2 > pos1 && pos2 > pos3 ){
			objParse.Class := tmpAry[4]
			if( pos1 > pos3 ){
				objParse.Path := tmpAry[3]
				objParse.ID := tmpAry[2]
			}else{
				objParse.Path := tmpAry[2]
				objParse.ID := tmpAry[3]
			}
		}else if( pos3 > pos1 && pos3 > pos2 ){
			objParse.ID := tmpAry[4]
			if( pos1 > pos2 ){
				objParse.Path := tmpAry[3]
				objParse.Class := tmpAry[2]
			}else{
				objParse.Path := tmpAry[2]
				objParse.Class := tmpAry[3]
			}
		}else{
			if( AHK_COMMON_LOG_INDEX ){
				LogOutput( "4つに分割されたパターン外エラー", AHK_COMMON_LOG_INDEX )
			}
			return -1
		}
	}
	if( tmpAry.Length = 3 ){
		if( pos1 > pos2 && pos3 = 0 ){
			objParse.Path := tmpAry[3]
			objParse.Class := tmpAry[2]
		}else if( pos2 > pos1 && pos3 = 0 ){
			objParse.Path := tmpAry[2]
			objParse.Class := tmpAry[3]
		}else if( pos1 > pos3 && pos2 = 0 ){
			objParse.Path := tmpAry[3]
			objParse.ID := Number(tmpAry[2])
		}else if( pos3 > pos1 && pos2 = 0 ){
			objParse.Path := tmpAry[2]
			objParse.ID := Number(tmpAry[3])
		}else if( pos2 > pos3 && pos1 = 0 ){
			objParse.Class := tmpAry[3]
			objParse.ID := Number(tmpAry[2])
		}else if( pos3 > pos2 && pos1 = 0 ){
			objParse.Class := tmpAry[2]
			objParse.ID := Number(tmpAry[3])
		}else{
			if( AHK_COMMON_LOG_INDEX ){
				LogOutput( "3つに分割されたパターン外エラー", AHK_COMMON_LOG_INDEX )
			}
			return -1
		}
	}
	if( tmpAry.Length = 2 ){
		if( pos2 = 0 && pos3 = 0 ){
			objParse.Path := tmpAry[2]
		}else if( pos1 = 0 && pos3 = 0 ){
			objParse.Class := tmpAry[2]
		}else if( pos1 = 0 && pos2 = 0 ){
			objParse.ID := Number(tmpAry[2])
		}else{
			if( AHK_COMMON_LOG_INDEX ){
				LogOutput( "2つに分割されたパターン外エラー", AHK_COMMON_LOG_INDEX )
			}
			return -1
		}
	}
	
	objParse.Title := Trim(objParse.Title)
	objParse.Path := Trim(objParse.Path)
	objParse.Class := Trim(objParse.Class)
	objParse.ID := Number(objParse.ID)

	if( AHK_COMMON_LOG_INDEX ){
		
		ttl := ""
		pth := ""
		cls := ""
		id := ""
		
		if( objParse.Title != "" ){
			ttl := " Title=" objParse.Title
		}
		if( objParse.Path != "" ){
			pth := " Path=" objParse.Path
		}
		if( objParse.Class != "" ){
			cls := " Class=" objParse.Class
		}
		if( objParse.ID != 0 ){
			id := " ID=" objParse.ID
		}
		LogOutput( "objParse: out2[" ttl pth cls id "]", AHK_COMMON_LOG_INDEX )
	}

	return 0
}


;//-----------------------------------------------------------------------------------------------------------------
;// PathToREGEX  未エスケープのパス文字列をRegEx対応の文字列に変換
;//  strPath   :   パス文字列
;//  戻り値    :   エスケープされたRegEx対応文字列
;//-----------------------------------------------------------------------------------------------------------------
PathToREGEX( strPath ){
	
	strTemp := strPath
	;//エスケープ済み
	if( InStr(strTemp,"C:\\") || InStr(strTemp,"\.") || InStr(strTemp,"\(") || InStr(strTemp,"\)") ){
		return strTemp
	}
	;//範囲指定のエスケープ不要処理済み
	if( InStr(strTemp,"\Q") && InStr(strTemp,"\E") ){
		return strTemp
	}
	
	;//ファイルパスとして使われる要エスケープ記号を\エスケープする
	if( InStr(strTemp,"\") ){
		strTemp := StrReplace(strTemp,"\","\\")
	}
	if( InStr(strTemp,"(") ){
		strTemp := StrReplace(strTemp,"(","\(")
	}
	if( InStr(strTemp,")") ){
		strTemp := StrReplace(strTemp,")","\)")
	}
	if( InStr(strTemp,".") ){
		strTemp := StrReplace(strTemp,".","\.")
	}
	return strTemp
	
}


;//-----------------------------------------------------------------------------------------------------------------
;// REGEXToPath  エスケープ済みのパス文字列を通常のパス文字列に変換
;//  strPath   :   パス文字列
;//  戻り値    :   エスケープを外したパス文字列
;//-----------------------------------------------------------------------------------------------------------------
REGEXToPath( strPath ){
	
	strTemp := strPath
	;//範囲指定のエスケープ不要処理済み
	if( InStr(strTemp,"\Q") && InStr(strTemp,"\E") ){
		strTemp := StrReplace(strTemp,"\Q","")
		strTemp := StrReplace(strTemp,"\E","")
		return strTemp
	}
	;//エスケープが無い
	if( ! InStr(strTemp,"\\") && ! InStr(strTemp,"\(") && ! InStr(strTemp,"\)") && ! InStr(strTemp,"\.") ){
		return strTemp
	}
	
	;//エスケープ記号を戻す
	if( InStr(strTemp,"\\") ){
		strTemp := StrReplace(strTemp,"\\","\")
	}
	if( InStr(strTemp,"\(") ){
		strTemp := StrReplace(strTemp,"\(","(")
	}
	if( InStr(strTemp,"\)") ){
		strTemp := StrReplace(strTemp,"\)",")")
	}
	if( InStr(strTemp,"\.") ){
		strTemp := StrReplace(strTemp,"\.",".")
	}
	return strTemp
	
}

;//-----------------------------------------------------------------------------------------------------------------
;// ConvertREGEX  ウィンドウ検索文字列を分析し未エスケープのパスとエスケープ済みのパスを相互変換
;//  strTarget  :   ウィンドウ検索文字列
;//  iMode      :   未エスケープからエスケープ済みに変換=0, エスケープ済みから未エスケープに変換=1
;//  戻り値     :   エスケープしたパス文字列で再生成したウィンドウ検索文字列
;//-----------------------------------------------------------------------------------------------------------------
ConvertREGEX( strTarget, iMode ){
	
	objParse := {}
	iRet := 0
	strRet := ""
	
	ttl := ""
	pth := ""
	cls := ""
	id := 0

	iRet := TargetParse( strTarget, &objParse )
	if( ! iRet ){
		
		if( objParse.Title != "" ){
			ttl := objParse.Title
		}
		if( objParse.Path != "" ){
			pth := objParse.Path
			if( ! iMode ){
				pth := PathToREGEX(pth)
			}else{
				pth := REGEXToPath(pth)
			}
		}
		if( objParse.Class != "" ){
			cls := objParse.Class
		}
		if( objParse.ID != 0 ){
			id := objParse.ID
		}
	}
	if( ttl != "" ){
		strRet := strRet ttl
	}
	if( pth != "" ){
		if( strRet != "" ){
			strRet := strRet " "
		}
		strRet := strRet "ahk_exe " pth
	}
	if( cls != "" ){
		if( strRet != "" ){
			strRet := strRet " "
		}
		strRet := strRet "ahk_class " cls
	}
	if( id != 0 ){
		if( strRet != "" ){
			strRet := strRet " "
		}
		strRet := strRet "ahk_id " id
	}
	
	return strRet

}


;//-----------------------------------------------------------------------------------------------------------------
;// TargetToREGEX  ウィンドウ検索文字列を分析し未エスケープのパスを変換
;//  strTarget  :   ウィンドウ検索文字列 (REGEX:タイトル ahk_exe パス ahk_class クラス ahk_id ハンドラ値)
;//  戻り値     :   エスケープしたパス文字列で再生成したウィンドウ検索文字列
;//-----------------------------------------------------------------------------------------------------------------
TargetToREGEX( strTarget ){
	
	strRet := ConvertREGEX( strTarget, 0 )
	if( AHK_COMMON_LOG_INDEX ){
		LogOutput( "TargetToREGEX: in [" strTarget "]", AHK_COMMON_LOG_INDEX )
		LogOutput( "TargetToREGEX: out[" strRet "]", AHK_COMMON_LOG_INDEX )
	}
	return strRet
}

;//-----------------------------------------------------------------------------------------------------------------
;// REGEXToTarget  ウィンドウ検索文字列を分析しエスケープ済みのパスを逆変換
;//  strTarget  :   ウィンドウ検索文字列 (REGEX:タイトル ahk_exe パス ahk_class クラス ahk_id ハンドラ値)
;//  戻り値     :   未エスケープに戻したパス文字列で再生成したウィンドウ検索文字列
;//-----------------------------------------------------------------------------------------------------------------
REGEXToTarget( strTarget ){

	strRet := ConvertREGEX( strTarget, 1 )
	if( AHK_COMMON_LOG_INDEX ){
		LogOutput( "REGEXToTarget: in [" strTarget "]", AHK_COMMON_LOG_INDEX )
		LogOutput( "REGEXToTarget: out[" strRet "]", AHK_COMMON_LOG_INDEX )
	}
	return strRet
}





;//カスタムツールチップ定義
global gA_CustomToolTipGui := Array()
global gA_CustomToolTipOpt := Array()
gA_CustomToolTipGui.Default := ""
gA_CustomToolTipOpt.Default := ""

;//-----------------------------------------------------------------------------------------------------------------
;// ShowCustomToolTip カスタムツールチップを表示する関数
;//  text      :  表示文字列(改行は"`n"), 無指定または""を指定すると破棄動作をする
;//  x         :  X座標(無指定にするとセンタリング, "C100"のように書くと左上ではなく中心座標)
;//  y         :  Y座標(無指定にするとセンタリング, "C150"のように書くと左上ではなく中心座標)
;//  obj       :  オブジェクトインデックス(1～, 省略時1)
;//  fore      :  前景色("RRGGBB", 省略時"000000")
;//  back      :  背景色("RRGGBB", 省略時"FFFFFF")
;//  font      :  フォント名(無指定や""はデフォルトから変更なし)
;//                 → https://ahkscript.github.io/ja/docs/v2/misc/FontsStandard.htm
;//                 サイズ指定やbold指定したい場合はフォント名の後ろに | で区切って記述する ex "BIZ UDGothic|bold s12"
;//                 フォントのオプション指定書式
;//                 → https://ahkscript.github.io/ja/docs/v2/lib/Gui.htm#SetFont
;//                 ※色指定は別の項目で設定しているのでここで指定しても上書きされてしまう
;//                 ※サイズだけ変更したい場合などはフォント名空欄で "|s10" のように記述します
;//  fixX      :  X座標が画面外にハミ出したら引き戻す=1, 何もしない=0 (省略時1)
;//  fixY      :  Y座標が画面外にハミ出したら引き戻す=1, 何もしない=0 (省略時1)
;//  clickClose:  クリックされたら閉じる形式のツールチップにする=1, クリックできない=0 (省略時0)
;//  戻り値    :  生成されたツールチップGUIのhWnd エラー時0
;// ※前回と同じパラメータを指定した場合は処理しない(チラ付き防止)仕様のため明示的に再表示する時は破棄してから使う
;// ※標準のToolTipと違い、座標指定CoordModeはScreen基準専用
;// ※クリックで閉じるモードで表示してもフェードアウト中はクリック判定が透過するので反応しません
;//-----------------------------------------------------------------------------------------------------------------
ShowCustomToolTip(text := "", x := "", y := "", obj := 1, fore := "000000", back := "FFFFFF", font := "", fixX := 1, fixY := 1, clickClose := 0 ) {

	global gA_CustomToolTipGui, gA_CustomToolTipOpt
	
	CUSTOMTOOLTIP_RETRY := 3 ;//リトライ最大回数
	CENTER_MODE := 99999
	centerX := 0
	centerY := 0
	posX := 0
	posY := 0
	getX := 0
	getY := 0
	getWidth := 0
	getHeight := 0
	
	fontOrg := font ;//元データを保持
	fontOpt := ""
	
	if( InStr(font,"|") ){
		aryTmp := StrSplit(font,"|")
		font := Trim(aryTmp[1])
		fontOpt := Trim(aryTmp[2])
	}
	
	;//共通関数内でCoordModeを使用する際、インクルードされるソースで別の指定がされていたら変わってしまうので
	;//待避して変更して使い終わったら戻す
	lastModeT := "" ;//ツールチップの座標系現在値を保持
	lastModeM := "" ;//メニューの座標系現在値を保持

	iRetry := 0

	;//オブジェクトインデックスが配列数を超えていたら拡張
	if( gA_CustomToolTipGui.length < obj ){
		gA_CustomToolTipGui.length := obj
		gA_CustomToolTipOpt.length := obj
		idx := 1
		while( obj >= idx ){
			if( gA_CustomToolTipOpt[idx] = "" ){
				gA_CustomToolTipOpt[idx] := Map("text","","x","","y","","fore","","back","","font","",
												"fixX","","fixY","","clickClose","")
			}
			idx++
		}
	}

	;//処理ループ中などで何度も同じToolTipテキストが指定された場合チラ付き防止にスキップする
	if( gA_CustomToolTipOpt[obj]["text"] = text &&
		gA_CustomToolTipOpt[obj]["x"] = x &&
		gA_CustomToolTipOpt[obj]["y"] = y &&
		gA_CustomToolTipOpt[obj]["fore"] = fore &&
		gA_CustomToolTipOpt[obj]["back"] = back &&
		gA_CustomToolTipOpt[obj]["font"] = fontOrg &&
		gA_CustomToolTipOpt[obj]["fixX"] = fixX &&
		gA_CustomToolTipOpt[obj]["fixY"] = fixY &&
		gA_CustomToolTipOpt[obj]["clickClose"] = clickClose
	  ){
		try{
			if( gA_CustomToolTipGui[obj].Hwnd != 0 ){
				return gA_CustomToolTipGui[obj].Hwnd
			}
		}
	}
	;//前回値を保持
	gA_CustomToolTipOpt[obj] := Map("text",text,"x",x,"y",y,"fore",fore,"back",back,"font",fontOrg,
									"fixX",fixX,"fixY",fixY,"clickClose",clickClose)

	if( text = "" ){
		DestroyCustomToolTip(obj)
		return 0
	}

	;//既に表示中であったら破棄
	try{
		gA_CustomToolTipGui[obj].Destroy()
	}

	;//センタリング指定
	if( x = "" ){
		x := CENTER_MODE
	}
	if( y = "" ){
		y := CENTER_MODE
	}

	;//中心座標指定だった場合数値部を抽出
	if( InStr(StrLower(x),"c") != 0 ){
		x := StrReplace(StrLower(x), "c", "")
		centerX := 1
	}
	if( InStr(StrLower(y),"c") != 0 ){
		y := StrReplace(StrLower(y), "c", "")
		centerY := 1
	}

	x := Number(x)
	y := Number(y)

	;//センタリング指定の場合
	if( x = CENTER_MODE ){
		posX := "xCenter"
	}else{
		posX := "x" x
	}
	if( y = CENTER_MODE ){
		posY := "yCenter"
	}else{
		posY := "y" y
	}

	Loop{
		;//生成直後のエラートラップ
		try{
			if( lastModeM = "" ){
				lastModeM := A_CoordModeMenu
			}
			CoordMode("Menu", "Screen")
			gA_CustomToolTipGui[obj] := Gui()
			
			if( font != "" || fontOpt != "" ){
				gA_CustomToolTipGui[obj].SetFont( fontOpt, font )
			}
			if( clickClose ){
				gA_CustomToolTipGui[obj].Opt("+AlwaysOnTop -Caption -SysMenu -ToolWindow +Owner")
			}else{
				gA_CustomToolTipGui[obj].Opt("+AlwaysOnTop +Disabled -Caption -SysMenu -ToolWindow +Owner")
			}
			mnuCtrlObj := gA_CustomToolTipGui[obj].Add("Text", "C" fore " vCtrlText", text)
			if( clickClose ){
				mnuCtrlObj.OnEvent("Click", OnClick_CustomToolTip)
				mnuCtrlObj.OnEvent("DoubleClick", OnDblClick_CustomToolTip)
				gA_CustomToolTipGui[obj].OnEvent("ContextMenu", OnRClick_CustomToolTip)
				gA_CustomToolTipGui[obj].OnEvent("Escape", OnEsc_CustomToolTip)
			}
			gA_CustomToolTipGui[obj].BackColor := back
			gA_CustomToolTipGui[obj].Show(posX " " posY " NoActivate AutoSize")
			gA_CustomToolTipGui[obj].GetPos(&getX, &getY, &getWidth, &getHeight)
			CoordMode("Menu", lastModeM)
			lastModeM := ""
			break
		}catch{
			if( lastModeM != "" ){
				CoordMode("Menu", lastModeM)
				lastModeM := ""
			}
			if( iRetry = CUSTOMTOOLTIP_RETRY ){
				return 0
			}
		}
		iRetry++
		Sleep(10 * iRetry)
	}

	;//中心座標指定の場合
	if( centerX ){
		getX := getX - getWidth / 2
	}
	if( centerY ){
		getY := getY - getHeight / 2
	}

	;//画面外に出ていたら引き戻す
	if( fixX = 1 ){
		if( getX < 0 ){
			getX := 0
		}
		if( getX + getWidth > A_ScreenWidth ){
			getX := A_ScreenWidth - getWidth
		}
	}
	if( fixY = 1 ){
		if( getY < 0 ){
			getY := 0
		}
		if( getY + getHeight > A_ScreenHeight ){
			getY := A_ScreenHeight - getHeight
		}
	}

	;//移動が必要な場合のみ移動
	if( (x != getX And x != CENTER_MODE) Or (y != getY And y != CENTER_MODE) ){
		;//生成後すぐに消滅してすれ違う事がある
		try{
			if( lastModeM = "" ){
				lastModeM := A_CoordModeMenu
			}
			CoordMode("Menu", "Screen")
			gA_CustomToolTipGui[obj].Move(getX, getY)
			CoordMode("Menu", lastModeM)
			lastModeM := ""
		}catch{
			if( lastModeM != "" ){
				CoordMode("Menu", lastModeM)
				lastModeM := ""
			}
		}
	}
	try{
		if( gA_CustomToolTipGui[obj].Hwnd != 0 ){
			return gA_CustomToolTipGui[obj].Hwnd
		}
	}
	return 0
}



;//-----------------------------------------------------------------------------------------------------------------
;// FadeoutCustomToolTip カスタムツールチップをフェードアウト後に消去する関数
;//  obj :        オブジェクトインデックス(1～, 省略時1)
;//  tim :        消去までのおおまかな時間ms(100～30000) ※処理オーバーヘッドに依存して時間が多少前後する
;//-----------------------------------------------------------------------------------------------------------------
FadeoutCustomToolTip( obj := 1, tim := 3000 ){
	
	global gA_CustomToolTipGui, gA_CustomToolTipOpt

	step := 255
	if( tim < 100 ){
		tim := 100
	}
	if( tim > 30000 ){
		tim := 30000
	}

	;LogInit("CustomToolTip")
	;LogOutput("tim=" tim)

	while(step){
		try{

			;LogOutput("STEP=" step)

			WinSetTransparent(Floor(step), "ahk_id " gA_CustomToolTipGui[obj].Hwnd)
			;//理論値では255step*100ms=25500だが処理時間ぶん差し引いてSleepを短めにしておく
			Sleep(75)
			;//フェード処理中に破棄されたら中断
			if( gA_CustomToolTipOpt[obj]["text"] = "" ){
				return
			}
			step := step - 25500/tim
			if( step < 0 ){
				step := 0
			}
			if( step = 0 ){
				DestroyCustomToolTip(obj)

				;LogOutput("Destroy")
			}
		}catch{
			return
		}
	}
}



;//-----------------------------------------------------------------------------------------------------------------
;// DestroyCustomToolTip カスタムツールチップを消去する関数
;//  obj :        オブジェクトインデックス(1～, 省略時1)
;//-----------------------------------------------------------------------------------------------------------------
DestroyCustomToolTip( obj := 1 ){

	global gA_CustomToolTipGui, gA_CustomToolTipOpt
	
	try{
		gA_CustomToolTipOpt[obj] := Map("text","","x","","y","","fore","","back","","font","",
										"fixX","","fixY","","clickClose","")
	}
	try{
		gA_CustomToolTipGui[obj].Destroy()
	}
	return 0
}


;//-----------------------------------------------------------------------------------------------------------------
;// OnClick_CustomToolTip       カスタムツールチップ左クリック時のイベント処理(GuiCtrl)
;//-----------------------------------------------------------------------------------------------------------------
OnClick_CustomToolTip(GuiCtrlObj, info){
	return OnLClick_CustomToolTip(GuiCtrlObj, info, 0)
}

;//-----------------------------------------------------------------------------------------------------------------
;// OnDblClick_CustomToolTip    カスタムツールチップ左ダブルクリック時のイベント処理(GuiCtrl)
;//-----------------------------------------------------------------------------------------------------------------
OnDblClick_CustomToolTip(GuiCtrlObj, info){
	return OnLClick_CustomToolTip(GuiCtrlObj, info, 1)
}

;//-----------------------------------------------------------------------------------------------------------------
;// OnLClick_CustomToolTip      カスタムツールチップの左クリック時の共通処理
;//  GuiCtrlObj :  メニューのコントロールオブジェクト
;//  info :        Textコントロールでは中身無効
;//  type :        左クリック=0, 左ダブルクリック=1
;//-----------------------------------------------------------------------------------------------------------------
OnLClick_CustomToolTip(GuiCtrlObj, info, type){

	global gA_CustomToolTipGui, gA_CustomToolTipOpt
	
	idx := 1
	while( idx <= gA_CustomToolTipGui.length ){
		try{
			if( GuiCtrlObj.Gui.Hwnd = gA_CustomToolTipGui[idx].Hwnd ){
				return OnEvent_CustomToolTip(idx, type)
			}
		}
		idx++
	}
	return -1
}

;//-----------------------------------------------------------------------------------------------------------------
;// OnRClick_CustomToolTip      カスタムツールチップ右クリック時のイベント処理(Gui)
;//-----------------------------------------------------------------------------------------------------------------
OnRClick_CustomToolTip(GuiObj, GuiCtrlObj, Item, IsRightClick, X, Y){

	global gA_CustomToolTipGui, gA_CustomToolTipOpt
	
	idx := 1
	while( idx <= gA_CustomToolTipGui.length ){
		try{
			if( GuiObj.Hwnd = gA_CustomToolTipGui[idx].Hwnd ){
				return OnEvent_CustomToolTip(idx, 2)
			}
		}
		idx++
	}
	return -1
}

;//-----------------------------------------------------------------------------------------------------------------
;// OnEsc_CustomToolTip        カスタムツールチップESC押下時のイベント処理(Gui)
;//-----------------------------------------------------------------------------------------------------------------
OnEsc_CustomToolTip(GuiObj){

	global gA_CustomToolTipGui, gA_CustomToolTipOpt
	
	idx := 1
	while( idx <= gA_CustomToolTipGui.length ){
		try{
			if( GuiObj.Hwnd = gA_CustomToolTipGui[idx].Hwnd ){
				return OnEvent_CustomToolTip(idx, 3)
			}
		}
		idx++
	}
	return -1
}

;//-----------------------------------------------------------------------------------------------------------------
;// OnEvent_CustomToolTip      カスタムツールチップのイベント処理
;//  obj  :        ツールチップオブジェクト配列の添字
;//  type :        左クリック=0, 左ダブルクリック=1, 右クリック=2, ESC=3
;//-----------------------------------------------------------------------------------------------------------------
OnEvent_CustomToolTip(obj, type){
	
	DestroyCustomToolTip(obj)
	return 0
}



/* 使用例

	;//消去(表示より先にやっておくことでタイマーのすれ違いを抑止する)
	SetTimer () => DestroyCustomToolTip(1), -3000

	;//表示
	ShowCustomToolTip("[CustomToolTip] Update", , , 1, "0000FF", "FFCF1F")
	ShowCustomToolTip("[CustomToolTip] Update", "C960", "C0", 1, "0000FF", "FFCF1F", 1, 0)
	ShowCustomToolTip("[CustomToolTip] Update", A_ScreenWidth, A_ScreenHeight, 1, "0000FF", "FFCF1F")
	ShowCustomToolTip("[CustomToolTip]`nUpdate", A_ScreenWidth, A_ScreenHeight, 1, "0000FF", "FFCF1F")
	ShowCustomToolTip("[CustomToolTip]`nUpdate", 320, 280, 1, "0000FF", "FFCF1F", "BIZ UDGothic|bold s12", 0, 0, 1)

	;//フェードアウト(待ち時間＋フェードアウト時間後に消去)
	SetTimer () => FadeoutCustomToolTip(1,2000), -1000

*/









;//ログ出力定義
global gA_LogOutput := Array()
gA_LogOutput.Default := ""

;//-----------------------------------------------------------------------------------------------------------------
;// LogInit ログ出力を行うための初期設定関数
;//  Title        : ログの分類名としてツール名などの文字列を設定
;//  FileName     : 出力先ファイル名
;//  FileDir      : 出力先フォルダ名(末尾は\なしで) ※省略時はexeまたはahkのカレント
;//  SJIS         : SJISで出力=1(デフォルト), UTF-8で出力=0
;//  dispMsgError : エラー発生時にMsgBox表示あり=1(デフォルト), なし=0
;//  logIndex     : ログ設定番号、省略時1  ※別のログファイルにも出力したい時に2以上を設定
;//                 共通関数自体のデバッグをする時には AHK_COMMON_LOG_INDEX に1以上を設定してLogInitする
;//  logMaxLine   : ログ保存最大行数、省略時0(無制限)
;//-----------------------------------------------------------------------------------------------------------------
LogInit( Title := "", FileName := "", FileDir := "", SJIS := 1, dispMsgError := 1, logIndex := 1, logMaxLine := 0 ) {

	global gA_LogOutput
	
	if( logIndex < 1 ){
		return
	}
	
	if( gA_LogOutput.length < logIndex ){
		gA_LogOutput.length := logIndex
		idx := 1
		While( logIndex >= idx ){
			if( gA_LogOutput[idx] = "" ){
				gA_LogOutput[idx] := Map("Path","","Title","","SJIS",0,"MsgError",0,"MaxLine",0)
			}
			idx++
		}
	}
	
	if( Title = "" ){
		gA_LogOutput[logIndex]["Title"] := "-----"
	}else{
		gA_LogOutput[logIndex]["Title"] := Title
	}
	if( FileDir = "" ){
		FileDir := A_ScriptDir
	}
	if( FileName = "" ){
		FileName := StrReplace( A_ScriptName,".exe","Log.txt" )
	}
	gA_LogOutput[logIndex]["Path"] := FileDir "\" FileName
	gA_LogOutput[logIndex]["SJIS"] := SJIS
	gA_LogOutput[logIndex]["MsgError"] := dispMsgError
	gA_LogOutput[logIndex]["MaxLine"] := logMaxLine
	
}

;//-----------------------------------------------------------------------------------------------------------------
;// LogOutput ログを出力する関数
;//  Text     : 出力文字列
;//  logIndex : ログ設定番号、省略時1(共通関数系が出力するのは1固定)別のログファイルにも出力したい時に2以上を設定
;//             対応するログ設定番号でLogInitが実行されてからしか使用できない
;//  fixedLog : 最大行数を指定している時に消さない行として指定=1, 通常ログ=0(省略時)
;//             起動直後の設定読み込みログなど残しておきたい部分を指定します
;//                ※固定部は先頭から続いている必要あり、固定部が終わって通常ログが書かれたらもう固定部を追加しない
;//                  事を制限事項とします。つまり固定部を書く使い方では起動毎にまずログを削除をする前提とします。
;//-----------------------------------------------------------------------------------------------------------------
LogOutput( Text := "", logIndex := 1, fixedLog := 0 ){

	global gA_LogOutput
	
	if( logIndex < 1 ){
		return
	}
	
	;//リトライテストをする時だけ1に設定
	RETRY_TEST := 0
	;//リトライテストにて意図的にファイルを排他したまま保持する時間
	RETRY_TEST_SLEEP := 50
	;//排他エラー時にリトライを試みる回数
	RETRY_MAX := 16
	
	if( gA_LogOutput.length < logIndex || gA_LogOutput[logIndex]["Path"] = "" ){
		MsgBox("ERROR: LogOutput[" logIndex "] is not initialized.`r`n[" Text "]")
		return
	}
	
	count := 0
	first := 1
	opt := ""
	crlf := "" ;//FileOpen～WriteLineではLFのみ自動的に付くのでCRLFにしたければテキスト末尾を "`r" とする
	fd := 0
	
	buf := Buffer(3) ;//DOM書き出し用バッファ
	NumPut( "UChar", 0xEF, "UChar", 0xBB, "UChar", 0xBF, buf) ;//UTF-8のBOMバイナリ3バイト
	;//本来は次の行が良いがByteToBuffからLogOutputする際に再帰呼び出しになってしまうのでここではNumPut直で書いておく
	;ByteToBuff( &buf, [ 0xEF, 0xBB, 0xBF ], 3 )
	
	debugStr := ""
	logMaxLine := gA_LogOutput[logIndex]["MaxLine"]
	
	if( gA_LogOutput[logIndex]["SJIS"] ){
		opt := "cp932"
		crlf := "`r" ;//WriteLineで書き込むとLFは付くのでテキスト末尾にCRを追加
	}else{
		if( logMaxLine = 0 ){
			opt := "UTF-8" ;//BOM付き(WriteLine関数のバグで"a"モードは良いが"rw"モードで余計な空行が先頭に入る)
		}else{
			opt := "UTF-8-RAW" ;//BOM無しモードにして手動で先頭にBOMを書き込む
		}
		crlf := "`r" ;//WriteLineで書き込むとLFは付くのでテキスト末尾にCRを追加
	}

	Loop{
		
		if( gA_LogOutput[logIndex]["MaxLine"] > 0 && fixedLog ){
			str := FormatTime(A_Now,"[yyyy/MM/dd HH:mm:ss]") "{" gA_LogOutput[logIndex]["Title"] "} " Text
		}else{
			str := FormatTime(A_Now,"[yyyy/MM/dd HH:mm:ss]") "[" gA_LogOutput[logIndex]["Title"] "] " Text
		}
		
		if( RETRY_TEST ){
			;//リトライのテスト
			if( count ){
				debugStr := " (LogRetry=" count ")"
			}
		}
		
		if( logMaxLine = 0 ){
			;//無条件追加
			try{
				;//排他オープン、追記
				if( ! fd ){
					fd := FileOpen( gA_LogOutput[logIndex]["Path"], "a-rwd", opt ) ;//"a"追記モード,"-rwd"読み書き削除排他
				}
				fd.WriteLine( str debugStr crlf )
				
				if( RETRY_TEST ){
					;//リトライのテスト
					Sleep(RETRY_TEST_SLEEP)
				}
				
				fd.Close()
				fd := 0
				break
				
			}catch{
				count++
				if( count = RETRY_MAX ){
					if( gA_LogOutput[logIndex]["MsgError"] ){
						MsgBox("ERROR: LogOutput RERTY OVER[" count "]")
					}
					break
				}
				Sleep(10 * count)
				continue
			}
			
		}else{
			;//最大行数以上の古い物から削除
			TextAry := Array()
			;//排他オープン、読み出しして追加して行数操作し上書きして新しいEOF位置で書き込み
			try{
				if( ! fd ){
					fd := FileOpen( gA_LogOutput[logIndex]["Path"], "rw-rwd", opt ) ;//"rw"更新モード,"-rwd"読み書き削除排他
				}
			}catch{
				count++
				if( count = RETRY_MAX ){
					if( gA_LogOutput[logIndex]["MsgError"] ){
						MsgBox("ERROR: LogOutput RERTY OVER[" count "]")
					}
					break
				}
				Sleep(10 * count)
				continue
			}
			try{
				;//先頭にシークして全て読み込む
				if( ! gA_LogOutput[logIndex]["SJIS"] ){
					fd.Seek( 0, 3 ) ;//UTF-8ではBOM部分を避けて読み込む
				}else{
					fd.Seek( 0, 0 )
				}
				while( ! fd.AtEOF ){
					tmpText := fd.ReadLine()
					TextAry.Push(tmpText)
				}
				
				;//追加されるログテキストを末尾に積んで余剰行数を先頭から削除
				TextAry.Push(str debugStr)
				if( TextAry.length > logMaxLine ){
					;//固定部を飛ばして可変部から消す
					iFixed := 1
					for(idx,tmp in TextAry){
						if( SubStr(tmp,22,1) = "{" ){
							iFixed++
						}else{
							break
						}
					}
					if( iFixed < logMaxLine ){
						TextAry.RemoveAt( iFixed, TextAry.length - logMaxLine )
					}else{
						if( gA_LogOutput[logIndex]["MsgError"] ){
							MsgBox("[" gA_LogOutput[logIndex]["Title"] "] 最大行数=" logMaxLine " 設定に対して`n"
								 . "固定ログがそれ以上書かれようとしました。`n"
								 . "ログ出力の最大行数設定を変更して下さい。")
						}
					}
				}
				;//先頭にシークして全て書き出す
				fd.Seek( 0, 0 )
				;//UTF-8は先頭にBOMを書き込む
				if( ! gA_LogOutput[logIndex]["SJIS"] ){
					fd.RawWrite( buf, 3 )
				}
				for(tmpText in TextAry){
					if( tmpText != "" ){
						fd.WriteLine( tmpText crlf )
					}
				}
				;//上書きすることで前回よりファイルの長さが短くなった場合ゴミが残るのでEOFを更新
				fd.Length := fd.Pos
				
				if( RETRY_TEST ){
					;//リトライのテスト
					Sleep(RETRY_TEST_SLEEP)
				}
				fd.Close()
				fd := 0
				break
				
			}catch{
				count++
				if( count = RETRY_MAX ){
					if( gA_LogOutput[logIndex]["MsgError"] ){
						MsgBox("ERROR: LogOutput RERTY OVER[" count "]")
					}
					break
				}
				Sleep(10 * count)
				continue
			}
		}
	}
	if( fd ){
		try{
			fd.Close()
		}
	}
}



;//-----------------------------------------------------------------------------------------------------------------
;// IsLogEnable  ログが初期化されているかどうかを返す
;//  logIndex : ログ設定番号、省略時1(共通関数系が出力するのは1固定)
;//   復帰値  : 初期化済みで有効=1, 無効=0
;//-----------------------------------------------------------------------------------------------------------------
IsLogEnable( logIndex := 1 ){

	global gA_LogOutput
	
	logPath := ""
	try{
		logPath := gA_LogOutput[logIndex]["Path"]
	}
	
	if( logPath != "" ){
		return 1
	}else{
		return 0
	}
}


/* 使用例

	LogInit("TestApp","TestLog.txt")
	LogOutput("Test App Start")

*/









;//########################################################################################################################################
;//※ウィンドウスタイル・拡張ウィンドウスタイルは基本的にWindowsAPIにてローレベルでウィンドウを作成する際に使用されるウィンドウの属性であり、
;//　起動して各種の初期化が済んでいるウィンドウに対して変更を行っても反映されないものがあります。動作が不安定になったり操作不能になる事も
;//　しばしばあるため変更するオペレーションは十分に注意して下さい。
;//########################################################################################################################################

/* ウィンドウスタイル定義 */
WS_POPUP              := 0x80000000 ;//ポップアップウィンドウを作成　(このスタイルはWS_CHILDスタイルと一緒には使えない)
WS_CHILD              := 0x40000000 ;//子ウィンドウを作成　(このスタイルはWS_POPUPスタイルと一緒には使えない)
WS_MINIMIZE           := 0x20000000 ;//ウィンドウを最小化の状態で作成
WS_VISIBLE            := 0x10000000 ;//可視状態のウィンドウを作成
WS_DISABLED           := 0x08000000 ;//無効(使用不能)なウィンドウを作成　(無効なウィンドウは有効にするまでユーザーからの入力を受け取らない)
WS_CLIPSIBLINGS       := 0x04000000 ;//兄弟関係にある子ウィンドウをクリップ
WS_CLIPCHILDREN       := 0x02000000 ;//親ウィンドウ内部を描画するときに子ウィンドウが占める領域を除外　(このスタイルは親ウィンドウを作成する時に使用)
WS_MAXIMIZE           := 0x01000000 ;//ウィンドウを最大化の状態で作成
WS_BORDER             := 0x00800000 ;//境界線を持つウィンドウを作成
WS_DLGFRAME           := 0x00400000 ;//ダイアログボックスで一般的に使われるスタイルの境界を持つウィンドウを作成
WS_VSCROLL            := 0x00200000 ;//垂直スクロールバーを持つウィンドウを作成
WS_HSCROLL            := 0x00100000 ;//水平スクロールバーを持つウィンドウを作成
WS_SYSMENU            := 0x00080000 ;//タイトルバー上にウィンドウメニューボックスを持つウィンドウを作成
WS_SIZEBOX            := 0x00040000 ;//サイズ変更境界を持つウィンドウを作成
WS_MINIMIZEBOX        := 0x00020000 ;//最小化ボタンを持つウィンドウを作成　(WS_SYSMENUも指定する必要あり、WS_EX_CONTEXTHELPを指定不可)
WS_MAXIMIZEBOX        := 0x00010000 ;//最大化ボタンを持つウィンドウを作成　(WS_SYSMENUも指定する必要あり、WS_EX_CONTEXTHELPを指定不可)
WS_OVERLAPPED         := 0x00000000 ;//オーバーラップウィンドウを作成　(オーバーラップウィンドウはタイトルと枠を持つ)

WS_CAPTION            := 0x00C00000 ;//タイトルバーを持つウィンドウを作成（WS_BORDER | WS_DLGFRAME）
WS_OVERLAPPEDWINDOW   := 0x00CF0000 ;//WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_THICKFRAME | WS_MINIMIZEBOX | WS_MAXIMIZEBOX
WS_POPUPWINDOW        := 0x80880000 ;//WS_POPUP | WS_BORDER | WS_SYSMENU

WS_BROWSER_ALLOW      := 0x16000000 ;//ブラウザ類の標準的なウィンドウを構成する必須フラグ(拙作ツール独自の定義)
WS_BROWSER_DENY       := 0x88000000 ;//ブラウザ類で非表示のウィンドウを構成する除外フラグ(拙作ツール独自の定義)

;// FirefoxESRの例       style=0x16C70000
;// FirefoxBetaの例      style=0x16CF0000
;// Chrmoeの例           style=0x16CF0000
;// Chrome最大化の例     style=0x17CF0000
;// Chrome最小化の例     style=0x36CF0000
;// Chromeフルスクリーン style=0x160B0000
;// Taskmgrの例          style=0x14CF0000
;// WindowSpyの例        style=0x14CE0000
;// ポップアップ系の例   style=0x94CA0000


/* 拡張ウィンドウスタイル定義 */
WS_EX_APPWINDOW       := 0x00040000 ;//ウィンドウが最小化されるとトップレベルウィンドウがタスクバー上に置かれる
WS_EX_STATICEDGE      := 0x00020000 ;//ユーザーの入力を受け付けない項目用の立体的に見える境界スタイルを持つウィンドウを作成
WS_EX_CONTROLPARENT   := 0x00010000 ;//ユーザーがTabキーを使って子ウィンドウ間を移動できるようにする
WS_EX_LEFTSCROLLBAR   := 0x00004000 ;//垂直スクロールバーがクライアント領域の左側に置く
WS_EX_RTLREADING      := 0x00002000 ;//右から左への読み取り順序を持つプロパティを持ったウィンドウを作成
WS_EX_RIGHT           := 0x00001000 ;//右揃えされたプロパティを持つウィンドウを作成
WS_EX_CONTEXTHELP     := 0x00000400 ;//ダイアログボックスのタイトルバーに［?］ボタンを追加
WS_EX_CLIENTEDGE      := 0x00000200 ;//縁が沈んで見える境界線を持つウィンドウを指定
WS_EX_WINDOWEDGE      := 0x00000100 ;//ウィンドウが盛り上がった縁の境界線を持つように指定
WS_EX_TOOLWINDOW      := 0x00000080 ;//ツールウィンドウを作成する
WS_EX_MDICHILD        := 0x00000040 ;//MDI 子ウィンドウを作成する
WS_EX_TRANSPARENT     := 0x00000020 ;//透過ウィンドウを作成する
WS_EX_ACCEPTFILES     := 0x00000010 ;//ドラッグアンドドロップでファイルを受け入れる
WS_EX_TOPMOST         := 0x00000008 ;//最前面ウィンドウを作成する
WS_EX_NOPARENTNOTIFY  := 0x00000004 ;//このスタイルで作成された子ウィンドウが作成/破棄された際、親ウィンドウにWM_PARENTNOTIFYを送らない
WS_EX_DLGMODALFRAME   := 0x00000001 ;//二重の境界線を持つウィンドウを作成する

WS_EX_OVERLAPPEDWINDOW:= 0x00000300 ;//WS_EX_WINDOWEDGE | WS_EX_CLIENTEDGE
WS_EX_PALETTEWINDOW   := 0x00000188 ;//WS_EX_WINDOWEDGE | WS_EX_TOOLWINDOW | WS_EX_TOPMOST


;//-----------------------------------------------------------------------------------------------------------------
;// StyleNumber ウィンドウスタイル定義名(拡張ウィンドウスタイル定義名)から定義値に変換する関数
;//  strStyle :   定義名文字列 WS_XXXX, WS_EX_YYYY または "WS_XXXX", "WS_EX_YYYY",
;//                            "+WS_SYSMENU", "-WS_SYSMENU", "^WS_SYSMENU", "WS_MINIMIZEBOX|WS_MAXIMIZEBOX"
;//                            WS_BROWSER_ALLOW/ALLOW, WS_BROWSER_DENY/DENY
;//  defStyle:    初期値(省略時0)
;//  戻り値 :     定義値 ※未定義の文字列が渡された場合は初期値を返す
;//-----------------------------------------------------------------------------------------------------------------
StyleNumber( strStyle, defStyle:=0x00000000 ){
	
	;//ダブルクォート付き、+指定、-指定, ^指定, |演算あり であるか
	useDQ := 0    ;// ex) "0x00000000"
	usePlus := 0  ;// ex) "+0x10000000"
	useMinus := 0 ;// ex) "-0x08000000"
	useHat := 0   ;// ex) "^0x88000000"
	usePipe := 0  ;// ex) "WS_MINIMIZEBOX|WS_MAXIMIZEBOX"
	
	strValue := 0
	strTemp := Trim(strStyle)
	
	;MsgBox("Start: strStyle=" strStyle)
	
	;//両端がダブルクォート
	if( SubStr(strTemp,1,1) = '"' && SubStr(strTemp,-1) = '"' ){
		useDQ := 1
		strTemp := StrReplace(strTemp,'"','')
	}
	;//先頭が +, -, ^
	headStr := SubStr(strTemp, 1, 1)
	if( headStr = '+' ){
		usePlus := 1
		strTemp := StrReplace(strTemp,'+','')
	}else if( headStr = '-' ){
		useMinus := 1
		strTemp := StrReplace(strTemp,'-','')
	}else if( headStr = '^' ){
		useHat := 1
		strTemp := StrReplace(strTemp,'^','')
	}
	;//文字列中のスペースを削除
	strTemp := StrReplace(strTemp,' ','')
	
	;//論理和演算付き
	if( InStr(strTemp, '|') > 0 ){
		aryTemp := StrSplit(strTemp,'|')
		iLoop := aryTemp.length
		;//文字列の数値化をしながらビットORを加算していく再帰呼び出し
		while( iLoop ){
			strValue := Number(strValue) | StrHex8(StyleNumber(aryTemp[iLoop]))
			iLoop--
		}
	}else{
		;//デファインから数値へ
		switch( StrUpper(strTemp) ){
		;//独自の定義
		case "WS_BROWSER_ALLOW", "ALLOW":
			strValue := WS_BROWSER_ALLOW
		case "WS_BROWSER_DENY", "DENY":
			strValue := WS_BROWSER_DENY
		;//ウィンドウスタイル定義
		case "WS_POPUP":
			strValue := WS_POPUP
		case "WS_CHILD":
			strValue := WS_CHILD
		case "WS_MINIMIZE":
			strValue := WS_MINIMIZE
		case "WS_VISIBLE":
			strValue := WS_VISIBLE
		case "WS_DISABLED":
			strValue := WS_DISABLED
		case "WS_CLIPSIBLINGS":
			strValue := WS_CLIPSIBLINGS
		case "WS_CLIPCHILDREN":
			strValue := WS_CLIPCHILDREN
		case "WS_MAXIMIZE":
			strValue := WS_MAXIMIZE
		case "WS_BORDER":
			strValue := WS_BORDER
		case "WS_DLGFRAME":
			strValue := WS_DLGFRAME
		case "WS_VSCROLL":
			strValue := WS_VSCROLL
		case "WS_HSCROLL":
			strValue := WS_HSCROLL
		case "WS_SYSMENU":
			strValue := WS_SYSMENU
		case "WS_SIZEBOX":
			strValue := WS_SIZEBOX
		case "WS_MINIMIZEBOX":
			strValue := WS_MINIMIZEBOX
		case "WS_MAXIMIZEBOX":
			strValue := WS_MAXIMIZEBOX
		case "WS_OVERLAPPED":
			strValue := WS_OVERLAPPED
		case "WS_CAPTION":
			strValue := WS_CAPTION
		case "WS_OVERLAPPEDWINDOW":
			strValue := WS_OVERLAPPEDWINDOW
		case "WS_POPUPWINDOW":
			strValue := WS_POPUPWINDOW
		;//拡張ウィンドウスタイル定義
		case "WS_EX_APPWINDOW":
			strValue := WS_EX_APPWINDOW
		case "WS_EX_STATICEDGE":
			strValue := WS_EX_STATICEDGE
		case "WS_EX_CONTROLPARENT":
			strValue := WS_EX_CONTROLPARENT
		case "WS_EX_LEFTSCROLLBAR":
			strValue := WS_EX_LEFTSCROLLBAR
		case "WS_EX_RTLREADING":
			strValue := WS_EX_RTLREADING
		case "WS_EX_RIGHT":
			strValue := WS_EX_RIGHT
		case "WS_EX_CONTEXTHELP":
			strValue := WS_EX_CONTEXTHELP
		case "WS_EX_CLIENTEDGE":
			strValue := WS_EX_CLIENTEDGE
		case "WS_EX_WINDOWEDGE":
			strValue := WS_EX_WINDOWEDGE
		case "WS_EX_TOOLWINDOW":
			strValue := WS_EX_TOOLWINDOW
		case "WS_EX_MDICHILD":
			strValue := WS_EX_MDICHILD
		case "WS_EX_TRANSPARENT":
			strValue := WS_EX_TRANSPARENT
		case "WS_EX_ACCEPTFILES":
			strValue := WS_EX_ACCEPTFILES
		case "WS_EX_TOPMOST":
			strValue := WS_EX_TOPMOST
		case "WS_EX_NOPARENTNOTIFY":
			strValue := WS_EX_NOPARENTNOTIFY
		case "WS_EX_DLGMODALFRAME":
			strValue := WS_EX_DLGMODALFRAME
		case "WS_EX_OVERLAPPEDWINDOW":
			strValue := WS_EX_OVERLAPPEDWINDOW
		case "WS_EX_PALETTEWINDOW":
			strValue := WS_EX_PALETTEWINDOW
		default:
			if( IsNumber(strTemp) ){
				strValue := Number(strTemp)
			}else{
				strValue := defStyle
			}
		}
	}
	
	;//数値化前に取り去った +, -, ^ " を戻す
	if( usePlus ){
		strValue := '+' StrHex8(strValue)
	}else if( useMinus ){
		strValue := '-' StrHex8(strValue)
	}else if( useHat ){
		strValue := '^' StrHex8(strValue)
	}else{
		strValue := StrHex8(strValue)
	}
	if( useDQ ){
		strValue := '"' strValue '"'
	}
	
	;MsgBox("Exit: strStyle=" strStyle " / strValue=" strValue)
	
	return strValue
}


;//-----------------------------------------------------------------------------------------------------------------
;// TestStyle  ウィンドウスタイルの必須ビット除外ビットにマッチするか判定
;//  strStyle :   判定するスタイル
;//  strAllow :   必須ビットマスク
;//  strDeny  :   除外ビットマスク
;//  hWnd     :   ウィンドウハンドルを渡すとログ出力(LogInitされている場合のみ) 省略時0でログ出力なし
;//               ログ番号の先頭インデックスに対して出力固定
;//  戻り値   :   マッチ=true, アンマッチ=false
;//-----------------------------------------------------------------------------------------------------------------
TestStyle( strStyle, strAllow, strDeny, hWnd:=0 ){
	
	fullPath := ""
	className := ""
	
	iLog := IsLogEnable(1) ;//共通関数自体のデバッグ意図ではなく一連の流れを記録したい場合 AHK_COMMON_LOG_INDEX ではなく通常ログに出す
	
	if( iLog && hWnd != 0 ){
		try{
			fullPath := WinGetProcessPath( "ahk_id " hWnd )
			className := WinGetClass( "ahk_id " hWnd )
		}catch{
			if( iLog ){
				LogOutput( "TestStyle: LostWindow: Style=" StrHex8(strStyle)
						 . " Allow=" StrHex8(strAllow) " Deny=" StrHex8(strDeny)
						 . " Path=" fullPath " Class=" className )
			}
		}
	}
	
	if( ! IsNumber(strStyle) || ! IsNumber(strAllow) || ! IsNumber(strDeny) ){
		if( iLog && hWnd != 0 ){
			LogOutput( "TestStyle: Param ERROR : Style=" strStyle
					 . " Allow=" strAllow " Deny=" strDeny " Path=" fullPath " Class=" className )
		}
		return false
	}
	if( (Number(strStyle) & Number(strAllow)) = Number(strAllow) && ! (Number(strStyle) & Number(strDeny)) ){
		if( iLog && hWnd != 0 ){
			LogOutput( "TestStyle: Match     : Style=" StrHex8(strStyle)
					 . " Allow=" StrHex8(strAllow) " Deny=" StrHex8(strDeny) " Path=" fullPath " Class=" className )
		}
		return true
	}
	if( iLog && hWnd != 0 ){
		LogOutput( "TestStyle: Not Match : Style=" StrHex8(strStyle)
				 . " Allow=" StrHex8(strAllow) " Deny=" StrHex8(strDeny) " Path=" fullPath " Class=" className )
	}
	return false
}


;//-----------------------------------------------------------------------------------------------------------------
;// GetWindowStyleStr ウィンドウスタイル定義名・定義値のUsage用文字列を返す
;//  戻り値 :     定義名・定義値の改行付き文字列を返す
;//-----------------------------------------------------------------------------------------------------------------
GetWindowStyleStr(){
	
	tmp := "`n"
	tmp := tmp " (WindowStyle definition)`n"
	tmp := tmp "  WS_POPUP              = 0x80000000`n"
	tmp := tmp "  WS_CHILD              = 0x40000000`n"
	tmp := tmp "  WS_MINIMIZE           = 0x20000000`n"
	tmp := tmp "  WS_VISIBLE            = 0x10000000`n"
	tmp := tmp "  WS_DISABLED           = 0x08000000`n"
	tmp := tmp "  WS_CLIPSIBLINGS       = 0x04000000`n"
	tmp := tmp "  WS_CLIPCHILDREN       = 0x02000000`n"
	tmp := tmp "  WS_MAXIMIZE           = 0x01000000`n"
	tmp := tmp "  WS_BORDER             = 0x00800000`n"
	tmp := tmp "  WS_DLGFRAME           = 0x00400000`n"
	tmp := tmp "  WS_VSCROLL            = 0x00200000`n"
	tmp := tmp "  WS_HSCROLL            = 0x00100000`n"
	tmp := tmp "  WS_SYSMENU            = 0x00080000`n"
	tmp := tmp "  WS_SIZEBOX            = 0x00040000`n"
	tmp := tmp "  WS_MINIMIZEBOX        = 0x00020000`n"
	tmp := tmp "  WS_MAXIMIZEBOX        = 0x00010000`n"
	tmp := tmp "  WS_CAPTION            = 0x00C00000`n"
	tmp := tmp "  WS_OVERLAPPEDWINDOW   = 0x00CF0000`n"
	tmp := tmp "  WS_POPUPWINDOW        = 0x80880000`n"
	tmp := tmp "  WS_BROWSER_ALLOW      = 0x16000000`n"
	tmp := tmp "  WS_BROWSER_DENY       = 0x88000000`n"
	
	return tmp
}


;//-----------------------------------------------------------------------------------------------------------------
;// GetWindowStyleExStr 拡張ウィンドウスタイル定義名・定義値のUsage用文字列を返す
;//  戻り値 :     定義名・定義値の改行付き文字列を返す
;//-----------------------------------------------------------------------------------------------------------------
GetWindowStyleExStr(){
	
	tmp := "`n"
	tmp := tmp " (WindowStyle definition)`n"
	tmp := tmp "  WS_EX_APPWINDOW       = 0x00040000`n"
	tmp := tmp "  WS_EX_STATICEDGE      = 0x00020000`n"
	tmp := tmp "  WS_EX_CONTROLPARENT   = 0x00010000`n"
	tmp := tmp "  WS_EX_LEFTSCROLLBAR   = 0x00004000`n"
	tmp := tmp "  WS_EX_RTLREADING      = 0x00002000`n"
	tmp := tmp "  WS_EX_RIGHT           = 0x00001000`n"
	tmp := tmp "  WS_EX_CONTEXTHELP     = 0x00000400`n"
	tmp := tmp "  WS_EX_CLIENTEDGE      = 0x00000200`n"
	tmp := tmp "  WS_EX_WINDOWEDGE      = 0x00000100`n"
	tmp := tmp "  WS_EX_TOOLWINDOW      = 0x00000080`n"
	tmp := tmp "  WS_EX_MDICHILD        = 0x00000040`n"
	tmp := tmp "  WS_EX_TRANSPARENT     = 0x00000020`n"
	tmp := tmp "  WS_EX_ACCEPTFILES     = 0x00000010`n"
	tmp := tmp "  WS_EX_TOPMOST         = 0x00000008`n"
	tmp := tmp "  WS_EX_NOPARENTNOTIFY  = 0x00000004`n"
	tmp := tmp "  WS_EX_DLGMODALFRAME   = 0x00000001`n"
	tmp := tmp "  WS_EX_OVERLAPPEDWINDOW= 0x00000300`n"
	tmp := tmp "  WS_EX_PALETTEWINDOW   = 0x00000188`n"
	
	return tmp
}





;//音声が再生中かどうか CheckAudioChange を繰り返し呼び出して取得した結果
global gi_audioPlaying := 0

;//-----------------------------------------------------------------------------------------------------------------
;// CheckAudioChange 音声が再生中かどうか返す
;//  threshold :  再生中と判定する音量値( 0.0～1.0 )
;//  戻り値    :  1=再生中 0=停止中
;// 出典 https://www.autohotkey.com/boards/viewtopic.php?style=19&t=134094
;//-----------------------------------------------------------------------------------------------------------------
CheckAudioChange( threshold := 0.0001 ){
	
	static audioMeter := ComValue(13, SoundGetInterface("{C02216F6-8C67-4B5B-9D00-D008E73E0064}")), peak := 0, playing := 0
	if( audioMeter ){
		ComCall( 3, audioMeter, "float*", &peak )
		if( peak > threshold ){
			if( playing = 1 ){
				return 1
			}
			playing := 1
			gi_audioPlaying := 1
		}else{
			if( playing = 0 ){
				return 0
			}
			playing := 0
			gi_audioPlaying := 0
		}
	}
	return playing
}



;//-----------------------------------------------------------------------------------------------------------------
;// SystemCursor マウスカーソルの変更
;//  cmd       :  "Show|Hide|Toggle|Reload"
;// 出典 https://ahkscript.github.io/ja/docs/v2/lib/DllCall.htm#ExHideCursor
;// ※終了時、元に戻すため OnExit (*) => SystemCursor("Show") を記述しておく必要あり
;//-----------------------------------------------------------------------------------------------------------------
SystemCursor(cmd)
{
	static visible := true, c := Map()
	static sys_cursors := [32512, 32513, 32514, 32515, 32516, 32642
						,  32643, 32644, 32645, 32646, 32648, 32649, 32650]
	if (cmd = "Reload" or !c.Count){ ;//リクエスト時またはファーストコール時にリロードします。
		for(i,id in sys_cursors){
			h_cursor  := DllCall("LoadCursor", "Ptr", 0, "Ptr", id)
			h_default := DllCall("CopyImage", "Ptr", h_cursor, "UInt", 2
								, "Int", 0, "Int", 0, "UInt", 0)
			h_blank   := DllCall("CreateCursor", "Ptr", 0, "Int", 0, "Int", 0
								, "Int", 32, "Int", 32
								, "Ptr", Buffer(32*4, 0xFF)
								, "Ptr", Buffer(32*4, 0))
			c[id] := {default: h_default, blank: h_blank}
		}
	}
	switch(cmd){
	case "Show": visible := true
	case "Hide": visible := false
	case "Toggle": visible := !visible
	default: return
	}
	for(id,handles in c){
		h_cursor := DllCall("CopyImage"
							, "Ptr", visible ? handles.default : handles.blank
							, "UInt", 2, "Int", 0, "Int", 0, "UInt", 0)
		DllCall("SetSystemCursor", "Ptr", h_cursor, "UInt", id)
	}
}





;//OnMessage, PostMessage/SendMessageで常駐アプリ間でプロセス間通信する時のMsgNumber
MSG_SYNC_PAUSE			:= 0x5963 ;//一時停止と再開のメッセージ番号
MSG_SYNC_START			:= 0x5964 ;//即時実行開始のメッセージ番号
;//OnMessage, PostMessage/SendMessageで常駐アプリ間でプロセス間通信する時のwParam(送信元・自身) ENUM値
APP_ACTIVEPSEUDO		:= 0x0001
APP_DISABLEMAXBUTTON	:= 0x0002
APP_RESIZEBROWSER		:= 0x0004
APP_SCREENSAVER			:= 0x0008
APP_UTILSMANAGER		:= 0x0010
;//一時停止と再開を同期する常駐アプリ列挙
global gObj_SyncPauseAppList := Map( APP_ACTIVEPSEUDO,     "ActivePseudo",
									 APP_DISABLEMAXBUTTON, "DisableMaxButton",
									 APP_RESIZEBROWSER,    "ResizeBrowser",
									 APP_SCREENSAVER,      "ScreenSaver",
									 APP_UTILSMANAGER,     "UtilsManager"
								   )
;//即時実行開始を同期する常駐アプリ列挙
global gObj_SyncStartAppList := Map( APP_ACTIVEPSEUDO,     "ActivePseudo",
									 APP_SCREENSAVER,      "ScreenSaver"
								   )

;//-----------------------------------------------------------------------------------------------------------------
;// SyncPause 停止再開を切り替え、他の常駐アプリに通知する
;//  iMyAppID  : 自身のスクリプトID (APP_ACTIVEPSEUDO, APP_DISABLEMAXBUTTON, APP_RESIZEBROWSER, APP_SCREENSAVER, APP_UTILSMANAGER)
;//  iPauseMode: OnMessageによるモード変更 再開=0, 一時停止=1, メニュー操作=-1
;//  iSyncPause: 各常駐アプリに通知を行う=1, 行わない=0
;//  defMenu   : タスクアイコンメニューのデフォルト項目
;//  戻り値    :  0=正常終了 負数=エラー
;//-----------------------------------------------------------------------------------------------------------------
SyncPause( iMyAppID, iPauseMode := -1, iSyncPause := 0, defMenu := ""){
	
	if( ! gObj_SyncPauseAppList.Has(iMyAppID) ){
		MsgBox("ERROR: SyncPause gObj_SyncPauseAppList[" iMyAppID "] is not defined.")
		return -1
	}
	strMyApp  := gObj_SyncPauseAppList[iMyAppID]
	iconName1 := strMyApp ".ico"
	iconName2 := strMyApp "2.ico"
	iLog := IsLogEnable(1) ;//共通関数自体のデバッグ意図ではなく一連の流れを記録したい場合 AHK_COMMON_LOG_INDEX ではなく通常ログに出す
	
	;//OnMessageから変更指示が来たが既に変更後の状態に等しい時は処理をしない
	if( iPauseMode = 0 ){
		try{
			A_TrayMenu.Rename("Continue and Restart", "Continue and Restart")
		}catch{
			return -2
		}
	}
	if( iPauseMode = 1 ){
		try{
			A_TrayMenu.Rename("Pause and Suspend", "Pause and Suspend")
		}catch{
			return -3
		}
	}
	
	if( (iPauseMode = -1 && (A_IsSuspended || A_IsPaused)) || iPauseMode = 0 ){
		;//一時停止
		isPause := 0
		try{
			A_TrayMenu.Rename("Continue and Restart", "Pause and Suspend")
			A_TrayMenu.Uncheck("Pause and Suspend")
			A_TrayMenu.Default := defMenu
		}catch{
			return -4
		}
		try{
			TraySetIcon(A_ScriptFullPath,1)
		}catch{
			TraySetIcon(iconName1)
		}
	}else{
		;//再開
		isPause := 1
		try{
			A_TrayMenu.Rename("Pause and Suspend", "Continue and Restart")
			A_TrayMenu.Check("Continue and Restart")
			A_TrayMenu.Default := "Continue and Restart"
		}catch{
			return -5
		}
		try{
			TraySetIcon(A_ScriptFullPath,2)
		}catch{
			TraySetIcon(iconName2)
		}
	}
	try{
		Pause(isPause)
		Suspend(isPause)
		if( iLog ){
			LogOutput("SyncPause:Pause and Suspend = " isPause)
		}
	}catch{
		return -6
	}
	
	if( iSyncPause && ! GetKeyState("Ctrl") ){
		;//自身で操作した場合のみ他のアプリにブロードキャストする
		dhw := A_DetectHiddenWindows
		DetectHiddenWindows True
		
		for(id, val in gObj_SyncPauseAppList){
			if( iMyAppID != id ){
				id := WinExist("ahk_exe " val ".exe ahk_class AutoHotkey")
				if( id > 0 ){
					try{
						iRet := SendMessage(MSG_SYNC_PAUSE, iMyAppID, isPause,, "ahk_id " id)
						if( iLog ){
							LogOutput("SyncPause:SendMessage Ret=" StrSpaceRight(iRet,4) " [" val "]")
						}
					}
				}
			}
		}
		DetectHiddenWindows dhw
	}
	return 0
}

;//-----------------------------------------------------------------------------------------------------------------
;// SyncStart   即時開始を実行し、他の常駐アプリに通知する
;//  iMyAppID  : 自身のスクリプトID (APP_ACTIVEPSEUDO, APP_DISABLEMAXBUTTON, APP_RESIZEBROWSER, APP_SCREENSAVER, APP_UTILSMANAGER)
;//  iSyncStart: 各常駐アプリに通知を行う=1, 行わない=0
;//  execFunc  : 即時実行を実施するパラメータ無しのコールバック関数
;//  戻り値    :  0=正常終了 負数=エラー
;//-----------------------------------------------------------------------------------------------------------------
SyncStart( iMyAppID, iSyncStart := 0, execFunc := "" ){

	if( ! gObj_SyncStartAppList.Has(iMyAppID) ){
		MsgBox("ERROR: SyncStart gObj_SyncStartAppList[" iMyAppID "] is not defined.")
		return -1
	}
	
	iLog := IsLogEnable(1) ;//共通関数自体のデバッグ意図ではなく一連の流れを記録したい場合 AHK_COMMON_LOG_INDEX ではなく通常ログに出す
	
	strMyApp  := gObj_SyncStartAppList[iMyAppID]
	
	if( iSyncStart && ! GetKeyState("Ctrl") ){
		;//自身で操作した場合のみ他のアプリにブロードキャストする
		dhw := A_DetectHiddenWindows
		DetectHiddenWindows True
		
		for(id, val in gObj_SyncStartAppList){
			if( iMyAppID != id ){
				id := WinExist("ahk_exe " val ".exe ahk_class AutoHotkey")
				if( id > 0 ){
					try{
						iRet := SendMessage(MSG_SYNC_START, iMyAppID, 0,, "ahk_id " id)
						if( iLog ){
							LogOutput("SyncStart:SendMessage Ret=" StrSpaceRight(iRet,4) " [" val "]")
						}
					}
				}
			}
		}
		DetectHiddenWindows dhw
	}
	if( execFunc != "" ){
		try{
			execFunc()
			if( iLog ){
				LogOutput("SyncStart:exec start")
			}
		}catch{
			return -2
		}
	}
	return 0

}



